#include "ingestor_client.h"

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/datetime/base.h>
#include <util/generic/size_literals.h>
#include <util/string/builder.h>
#include <util/system/hostname.h>

namespace NSolomon::NIngestor {
namespace {
    using namespace NMonitoring;
    using namespace NSolomon;
    using namespace NThreading;
    using namespace grpc;
    using namespace yandex::monitoring::ingestor;

    using yandex::solomon::config::rpc::TGrpcClientConfig;

    using NGrpc::TGRpcClientLow;
    using NGrpc::TGRpcClientConfig;
    using NGrpc::TGrpcStatus;
    using NGrpc::TServiceConnection;

    TShardCreated ConvertShardCreatedFromProto(TCreateShardResponse&& resp) {
        TShardCreated result;
        result.ShardId = resp.GetShardId();
        result.Leader = resp.GetLeaderHost();
        result.NumId = resp.GetNumId();
        result.AssignedToHost = resp.GetAssignedToHost();

        return result;
    }

    class TGrpcIngestorClientBase: public IIngestorClient {
    public:
        TGrpcIngestorClientBase(
                IMetricRegistry&,
                std::unique_ptr<TGrpcServiceConnection<IngestorService>> connection)
            : Connection_{std::move(connection)}
        {
        }

        TAsyncDataProcessResponse ProcessPulledData(const TPulledDataRequest& request) noexcept override {
            auto promise = NewPromise<TErrorOr<TDataProcessResponse, TApiCallError>>();

            auto cb = [promise] (TGrpcStatus&& status, TDataProcessResponse&& result) mutable {
                if (!status.Ok()) {
                    promise.SetValue(FromGrpcError<TDataProcessResponse>(status));
                } else {
                    promise.SetValue(std::move(result));
                }
            };

            Connection_->Request<TPulledDataRequest, TDataProcessResponse>(
                request,
                std::move(cb),
                &IngestorService::Stub::AsyncProcessPulledData);

            return promise.GetFuture();
        }

        TAsyncDataProcessResponse ProcessPushedData(const TPushedDataRequest& request) noexcept override {
            auto promise = NewPromise<TErrorOr<TDataProcessResponse, TApiCallError>>();

            auto cb = [promise] (TGrpcStatus&& status, TDataProcessResponse&& result) mutable {
                if (!status.Ok()) {
                    promise.SetValue(FromGrpcError<TDataProcessResponse>(status));
                } else {
                    promise.SetValue(std::move(result));
                }
            };

            Connection_->Request<TPushedDataRequest, TDataProcessResponse>(
                request,
                std::move(cb),
                &IngestorService::Stub::AsyncProcessPushedData);

            return promise.GetFuture();
        }

        TAsyncGetAssignedShardsResponse GetAssignedShards(const TGetAssignedShardsRequest& request) noexcept override {
            auto promise = NewPromise<TGetAssignedShardsResponseOrErrorPtr>();

            auto cb = [promise] (TGrpcStatus&& status, TGetAssignedShardsResponse&& result) mutable {
                if (!status.Ok()) {
                    promise.SetValue(std::make_unique<TGetAssignedShardsResponseOrError>(status));
                } else {
                    promise.SetValue(std::make_unique<TGetAssignedShardsResponseOrError>(std::move(result)));
                }
            };

            Connection_->Request<TGetAssignedShardsRequest, TGetAssignedShardsResponse>(
                request,
                std::move(cb),
                &IngestorService::Stub::AsyncGetAssignedShards);

            return promise;
        };

        TAsyncShardCreated CreateShard(const TCreateShardRequest& request) noexcept override {
            auto promise = NewPromise<TErrorOr<TShardCreated, TApiCallError>>();

            auto cb = [promise] (TGrpcStatus&& status, TCreateShardResponse&& result) mutable {
                if (!status.Ok()) {
                    promise.SetValue(FromGrpcError<TShardCreated>(status));
                } else {
                    promise.SetValue(ConvertShardCreatedFromProto(std::move(result)));
                }
            };

            Connection_->Request<TCreateShardRequest, TCreateShardResponse>(
                request,
                std::move(cb),
                &IngestorService::Stub::AsyncCreateShard);

            return promise;
        }

        void Stop(bool) noexcept override {
        }

    private:
        std::unique_ptr<TGrpcServiceConnection<IngestorService>> Connection_;
    };

    IIngestorClientPtr CreateIngestorClient(
            IMetricRegistry& registry,
            std::unique_ptr<TGrpcServiceConnection<IngestorService>> connection)
    {
        return std::make_unique<TGrpcIngestorClientBase>(registry, std::move(connection));
    }
} // namespace

IIngestorClientPtr CreateIngestorGrpcClient(const TGrpcClientConfig& conf, TMetricRegistry& registry, TString clientId) {
    auto threadPool = CreateGrpcThreadPool(conf);
    auto sc = CreateGrpcServiceConnection<IngestorService>(
        conf,
        false,
        registry,
        std::move(threadPool),
        std::move(clientId));

    return CreateIngestorClient(registry, std::move(sc));
}

IIngestorClusterClientPtr CreateIngestorGrpcClusterClient(const TGrpcClientConfig& conf, TMetricRegistry& registry, TString clientId) {
    return std::make_shared<TGrpcClusterClientBase<IngestorService, IIngestorClient>>(
        conf,
        false,
        registry,
        CreateIngestorClient,
        std::move(clientId));
}

} // namespace NSolomon::NIngestor
