#include "rpc.h"

#include <ydb/core/protos/grpc.grpc.pb.h>
#include <ydb/public/lib/base/msgbus_status.h>

#include <util/generic/hash.h>
#include <util/random/random.h>

namespace NSolomon::NKikimr {

using yandex::solomon::config::rpc::TGrpcClientConfig;
using namespace NKikimrClient;

namespace {

class TKikimrMetrics {
public:
    explicit TKikimrMetrics(NMonitoring::IMetricRegistry& metrics) {
        NMonitoring::TLabel rps{"sensor", "kv.client.rps"};
        PerStatus_[METRIC_OK] = metrics.Rate(NMonitoring::MakeLabels({rps, {"status", "OK"}}));
        PerStatus_[METRIC_KV_ERROR] = metrics.Rate(NMonitoring::MakeLabels({rps, {"status", "KV_ERROR"}}));
        PerStatus_[METRIC_KV_TIMEOUT] = metrics.Rate(NMonitoring::MakeLabels({rps, {"status", "KV_TIMEOUT"}}));
        PerStatus_[METRIC_KV_INTERNAL] = metrics.Rate(NMonitoring::MakeLabels({rps, {"status", "KV_INTERNAL"}}));
        PerStatus_[METRIC_GRPC_TIMEOUT] = metrics.Rate(NMonitoring::MakeLabels({rps, {"status", "GRPC_TIMEOUT"}}));
        PerStatus_[METRIC_GRPC_RESOURCE_EXHAUSTED] = metrics.Rate(NMonitoring::MakeLabels({rps, {"status", "GRPC_RESOURCE_EXHAUSTED"}}));
        PerStatus_[METRIC_GRPC_INTERNAL] = metrics.Rate(NMonitoring::MakeLabels({rps, {"status", "GRPC_INTERNAL"}}));
        PerStatus_[METRIC_GRPC_UNAVAILABLE] = metrics.Rate(NMonitoring::MakeLabels({rps, {"status", "GRPC_UNAVAILABLE"}}));
        PerStatus_[METRIC_OTHER] = metrics.Rate(NMonitoring::MakeLabels({rps, {"status", "OTHER"}}));

        InboundDataRate_ = metrics.Rate(NMonitoring::MakeLabels({{"sensor", "kv.client.inboundDataRate"}}));
        OutboundDataRate_ = metrics.Rate(NMonitoring::MakeLabels({{"sensor", "kv.client.outboundDataRate"}}));
        InFlightMetric_ = metrics.IntGauge(NMonitoring::MakeLabels({{"sensor", "kv.client.inflight"}}));
        InFlightBytesMetric_ = metrics.IntGauge(NMonitoring::MakeLabels({{"sensor", "kv.client.inflightBytes"}}));
        ResponseTime_ = metrics.HistogramRate(
                NMonitoring::MakeLabels({{"sensor", "kv.client.elapsedTimeMs"}}),
                NMonitoring::ExponentialHistogram(16, 2, 1));
    }

    void RecordOutbound(size_t reqSize) {
        OutboundDataRate_->Add(reqSize);
        InFlightMetric_->Add(1);
        InFlightBytesMetric_->Add(static_cast<i64>(reqSize));
    }

    void RecordInbound(NMonitoring::IRate* statusRate, size_t reqSize, size_t respSize, TDuration time) {
        statusRate->Inc();
        InFlightMetric_->Add(-1);
        InFlightBytesMetric_->Add(-static_cast<i64>(reqSize));
        InboundDataRate_->Add(respSize);
        ResponseTime_->Record(static_cast<i64>(time.MilliSeconds()));
    }

    void RecordSuccess(size_t size, size_t responseSize, TDuration time, ::NKikimr::NMsgBusProxy::EResponseStatus status) {
        auto metric = PerStatus_[METRIC_OTHER];
        switch (status) {
            case ::NKikimr::NMsgBusProxy::MSTATUS_OK:
                metric = PerStatus_[METRIC_OK];
                break;
            case ::NKikimr::NMsgBusProxy::MSTATUS_ERROR:
                metric = PerStatus_[METRIC_KV_ERROR];
                break;
            case ::NKikimr::NMsgBusProxy::MSTATUS_TIMEOUT:
                metric = PerStatus_[METRIC_KV_TIMEOUT];
                break;
            case ::NKikimr::NMsgBusProxy::MSTATUS_INTERNALERROR:
                metric = PerStatus_[METRIC_KV_INTERNAL];
                break;
            default:
                break;
        }

        RecordInbound(metric, size, responseSize, time);
    }

    void RecordError(size_t size, TDuration time, grpc::StatusCode status) {
        auto metric = PerStatus_[METRIC_OTHER];
        switch (status) {
            case grpc::DEADLINE_EXCEEDED:
                metric = PerStatus_[METRIC_GRPC_TIMEOUT];
                break;
            case grpc::RESOURCE_EXHAUSTED:
                metric = PerStatus_[METRIC_GRPC_RESOURCE_EXHAUSTED];
                break;
            case grpc::INTERNAL:
                metric = PerStatus_[METRIC_GRPC_INTERNAL];
                break;
            case grpc::UNAVAILABLE:
                metric = PerStatus_[METRIC_GRPC_UNAVAILABLE];
                break;
            default:
                break;
        }

        RecordInbound(metric, size, 0, time);
    }

private:
    enum ERateMetric {
        METRIC_OK,
        METRIC_KV_ERROR,
        METRIC_KV_TIMEOUT,
        METRIC_KV_INTERNAL,
        METRIC_GRPC_TIMEOUT,
        METRIC_GRPC_RESOURCE_EXHAUSTED,
        METRIC_GRPC_INTERNAL,
        METRIC_GRPC_UNAVAILABLE,
        METRIC_OTHER,

        NUM_OF_METRICS
    };

    std::array<NMonitoring::IRate*, NUM_OF_METRICS> PerStatus_{};
    NMonitoring::IRate* InboundDataRate_ = nullptr;
    NMonitoring::IRate* OutboundDataRate_ = nullptr;
    NMonitoring::IIntGauge* InFlightMetric_ = nullptr;
    NMonitoring::IIntGauge* InFlightBytesMetric_ = nullptr;
    NMonitoring::IHistogram* ResponseTime_ = nullptr;
};

class TKikimrRpcImpl final: public IKikimrRpc {
public:
    TKikimrRpcImpl(
            NMonitoring::IMetricRegistry& metrics,
            std::unique_ptr<TGrpcServiceConnection<TGRpcServer>> connection) noexcept
        : Metrics_{metrics}
        , Connection_{std::move(connection)}
    {
    }

    TKikimrAsyncResponse SchemeDescribe(const TSchemeDescribe& req) override {
        return Request<TSchemeDescribe, TResponse, &TGRpcServer::Stub::AsyncSchemeDescribe>(req);
    }

    TKikimrAsyncResponse SchemeOperation(const TSchemeOperation& req) override {
        return Request<TSchemeOperation, TResponse, &TGRpcServer::Stub::AsyncSchemeOperation>(req);
    }

    TKikimrAsyncResponse SchemeOperationStatus(const TSchemeOperationStatus& req) override {
        return Request<TSchemeOperationStatus, TResponse, &TGRpcServer::Stub::AsyncSchemeOperationStatus>(req);
    }

    TKikimrAsyncResponse HiveCreateTablet(const THiveCreateTablet& req) override {
        return Request<THiveCreateTablet, TResponse, &TGRpcServer::Stub::AsyncHiveCreateTablet>(req);
    }

    TKikimrAsyncResponse TabletStateRequest(const TTabletStateRequest& req) override {
        return Request<TTabletStateRequest, TResponse, &TGRpcServer::Stub::AsyncTabletStateRequest>(req);
    }

    TKikimrAsyncResponse LocalEnumerateTablets(const TLocalEnumerateTablets& req) override {
        return Request<TLocalEnumerateTablets, TResponse, &TGRpcServer::Stub::AsyncLocalEnumerateTablets>(req);
    }

    TKikimrAsyncResponse KeyValue(const TKeyValueRequest& req) override {
        return Request<TKeyValueRequest, TResponse, &TGRpcServer::Stub::AsyncKeyValue>(req);
    }

    void Stop(bool) override {
        // nop
    }

private:
    template <typename TReq, typename TRes, auto Request>
    NThreading::TFuture<TErrorOr<TRes, NGrpc::TGrpcStatus>> Request(const TReq& request) {
        size_t reqSize = request.ByteSizeLong();
        Metrics_.RecordOutbound(reqSize);
        auto start = TInstant::Now();

        auto promise = NThreading::NewPromise<TErrorOr<TRes, NGrpc::TGrpcStatus>>();
        auto cb = [promise, reqSize, start, this] (NGrpc::TGrpcStatus&& status, TRes&& result) mutable {
            if (status.Ok()) {
                auto resStatus = static_cast<::NKikimr::NMsgBusProxy::EResponseStatus>(result.GetStatus());
                Metrics_.RecordSuccess(reqSize, result.ByteSizeLong(), TInstant::Now() - start, resStatus);
                promise.SetValue(std::move(result));
            } else {
                auto resStatus = static_cast<grpc::StatusCode>(status.GRpcStatusCode);
                Metrics_.RecordError(reqSize, TInstant::Now() - start, resStatus);
                promise.SetValue(std::move(status));
            }
        };

        Connection_->Request<TReq, TRes>(request, std::move(cb), Request);
        return promise.GetFuture();
    }

private:
    TKikimrMetrics Metrics_;
    std::unique_ptr<TGrpcServiceConnection<TGRpcServer>> Connection_;
};

std::unique_ptr<IKikimrRpc> CreateKikimrRpc(
        NMonitoring::IMetricRegistry& metrics,
        std::unique_ptr<TGrpcServiceConnection<TGRpcServer>> connection)
{
    return std::make_unique<TKikimrRpcImpl>(metrics, std::move(connection));
}

} // namespace

std::unique_ptr<IKikimrRpc> CreateNodeGrpc(
        const yandex::solomon::config::rpc::TGrpcClientConfig& conf,
        NMonitoring::IMetricRegistry& registry,
        TString clientId)
{
    auto threadPool = CreateGrpcThreadPool(conf);
    auto sc = CreateGrpcServiceConnection<TGRpcServer>(conf, false, registry, std::move(threadPool), std::move(clientId));

    return CreateKikimrRpc(registry, std::move(sc));
}

std::shared_ptr<IKikimrClusterRpc> CreateClusterGrpc(
        const yandex::solomon::config::rpc::TGrpcClientConfig& conf,
        NMonitoring::IMetricRegistry& registry,
        TString clientId)
{
    return std::make_shared<TGrpcClusterClientBase<TGRpcServer, IKikimrRpc>>(
        conf,
        false,
        registry,
        CreateKikimrRpc,
        std::move(clientId));
}

} // namespace NSolomon::NKikimr
