#pragma once

#include "rpc.h"

#include <util/random/random.h>

namespace NSolomon::NKikimr {

/**
 * Stub implementation of Kikimr RPC.
 */
class TKikimrRpcStub: public IKikimrRpc {
public:
    TKikimrAsyncResponse SchemeDescribe(const NKikimrClient::TSchemeDescribe&) override {
        return NotImplemented<TKikimrAsyncResponse>("SchemeDescribe");
    }

    TKikimrAsyncResponse SchemeOperation(const NKikimrClient::TSchemeOperation&) override {
        return NotImplemented<TKikimrAsyncResponse>("SchemeOperation");
    }

    TKikimrAsyncResponse SchemeOperationStatus(const NKikimrClient::TSchemeOperationStatus&) override {
        return NotImplemented<TKikimrAsyncResponse>("SchemeOperationStatus");
    }

    TKikimrAsyncResponse HiveCreateTablet(const NKikimrClient::THiveCreateTablet&) override {
        return NotImplemented<TKikimrAsyncResponse>("HiveCreateTablet");
    }

    TKikimrAsyncResponse TabletStateRequest(const NKikimrClient::TTabletStateRequest&) override {
        return NotImplemented<TKikimrAsyncResponse>("TabletStateRequest");
    }

    TKikimrAsyncResponse LocalEnumerateTablets(const NKikimrClient::TLocalEnumerateTablets&) override {
        return NotImplemented<TKikimrAsyncResponse>("LocalEnumerateTablets");
    }

    TKikimrAsyncResponse KeyValue(const NKikimrClient::TKeyValueRequest&) override {
        return NotImplemented<TKikimrAsyncResponse>("KeyValue");
    }

    void Stop(bool /*wait*/) override {
        // nop
    }

protected:
    template <typename T>
    static T NotImplemented(TStringBuf method) {
        return NThreading::MakeErrorFuture<NThreading::TFutureType<T>>(
                std::make_exception_ptr(yexception() << "KikimrRpc::" << method << "() is not implemented"));
    }

    template <typename T>
    static T GrpcError(grpc::StatusCode statusCode, TString message) {
        using U = NThreading::TFutureType<T>;
        return NThreading::MakeFuture<U>(U::FromError(NGrpc::TGrpcStatus(statusCode, std::move(message))));
    }

    template <typename T>
    static T Fail(TStringBuf message) {
        using U = NThreading::TFutureType<T>;
        return NThreading::MakeErrorFuture<U>(std::make_exception_ptr(yexception() << message));
    }
};

/**
 * Stub implementation of Kikimr cluster RPC.
 */
class TKikimrClusterRpcStub: public IClusterRpc<IKikimrRpc> {
public:
    explicit TKikimrClusterRpcStub(THashMap<TString, std::unique_ptr<IKikimrRpc>> nodeRpc)
        : NodeRpc_{std::move(nodeRpc)}
    {
        Addresses_.reserve(NodeRpc_.size());
        for (const auto& [k, v]: NodeRpc_) {
            Addresses_.emplace_back(k);
        }
    }

    IKikimrRpc* Get(TStringBuf address) noexcept override {
        auto it = NodeRpc_.find(address);
        return it == NodeRpc_.end() ? nullptr : it->second.get();
    }

    IKikimrRpc* GetAny() noexcept override {
        auto idx = RandomNumber<size_t>(Addresses_.size());
        return Get(Addresses_[idx]);
    }

    const std::vector<TString>& Addresses() const noexcept override {
        return Addresses_;
    }

    void Add(TStringBuf) override {
        ythrow yexception() << "not implemented";
    }

    void Stop(bool /*wait*/) override {
        // nop
    }

private:
    THashMap<TString, std::unique_ptr<IKikimrRpc>> NodeRpc_;
    std::vector<TString> Addresses_;
};

} // namespace NSolomon::NKikimr
