#include <solomon/libs/cpp/clients/load_info/load_service_client.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/getopt/modchooser.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/stream/file.h>
#include <util/folder/path.h>

using namespace NLastGetopt;
using namespace NSolomon;
using namespace NLoadInfo;

NMonitoring::TMetricRegistry DUMMY_REGISTRY;

class TCommandBase: public TMainClassArgs {
protected:
    virtual void DoRegisterOptions(NLastGetopt::TOpts& opts) = 0;
    virtual int DoExecute(NLastGetopt::TOptsParseResult&& opts, ILoadInfoClientPtr loadInfoClient) = 0;

private:
    void RegisterOptions(NLastGetopt::TOpts& opts) override {
        opts.AddLongOption('p', "port", "Node port to connect to")
                .RequiredArgument("PORT")
                .StoreResult(&Port_)
                .DefaultValue(4760);
        opts.AddLongOption('h', "host", "Node host to connect to")
                .RequiredArgument("HOST")
                .StoreResult(&Host_)
                .DefaultValue("localhost");

        DoRegisterOptions(opts);
    }

    int DoRun(NLastGetopt::TOptsParseResult&& opts) override {
        yandex::solomon::config::rpc::TGrpcClientConfig conf;

        auto* addr = conf.AddAddresses();
        *addr = TStringBuilder() << Host_ << ':' << Port_;

        return DoExecute(
            std::move(opts),
            CreateLoadInfoGrpcClient(conf, DUMMY_REGISTRY)
        );
    }

protected:
    TString Host_;
    ui16 Port_;
};

class TGetLoadInfoCommand: public TCommandBase {
    void DoRegisterOptions(NLastGetopt::TOpts& opts) override {
        opts.AddLongOption("address", "address load info is requested for")
                .RequiredArgument("HOST");
    }

    int DoExecute(NLastGetopt::TOptsParseResult&& opts, ILoadInfoClientPtr loadInfoClient) override {
        auto address = opts.Get("address");
        const auto& loadInfoOrError = loadInfoClient->GetLoadInfo(address).ExtractValueSync();

        if (!loadInfoOrError.Success()) {
            Cerr << "failed to get load info: " << loadInfoOrError.Error().Message() << Endl;
            return 1;
        }

        auto& val = loadInfoOrError.Value();
        auto& shards = val.shards_load();

        for (int i = 0; i < val.shards_load().shard_ids_size(); ++i) {
            Cout << "shard:" << Endl;
            Cout << "\t" << "id: " << shards.shard_ids(i) << Endl;
            // TODO(ivanzhukov@):
//            Cout << "\t" << "state: " << shards.states(i) << Endl;
//            Cout << "\t" << "uptime: " << shards.uptime_millis(i) << "ms" << Endl;
            Cout << "\t" << "cpu time: " << shards.cpu_time_nanos(i) << "ns" << Endl;
            Cout << "\t" << "memory: " << shards.memory_bytes(i) << "b" << Endl;
            Cout << "\t" << "network: " << shards.network_bytes(i) << "b" << Endl;
        }

        return 0;
    }
};

template <class TCommand>
TMainClassArgs* GetCommand() {
    return Singleton<TCommand>();
}

int Main(int argc, const char** argv) {
    NLastGetopt::NComp::TCustomCompleter::FireCustomCompleter(argc, argv);

    TModChooser modeChooser;
    modeChooser.SetDescription("load-info -- a console client for the LoadInfo gRPC service");
    modeChooser.SetPrintShortCommandInUsage(false);
    modeChooser.SetModesHelpOption("-h");

    modeChooser.AddGroupModeDescription("gRPC commands", false);
    modeChooser.AddMode(
            "get-load-info",
            GetCommand<TGetLoadInfoCommand>(),
            "Prints load info of a host",
            false,
            false);

    modeChooser.AddGroupModeDescription("internal operations", false);
    modeChooser.AddCompletions("load-info", "completion", false, true);

    return modeChooser.Run(argc, argv);
};

int main(int argc, const char** argv) {
    try {
        return Main(argc, argv);
    } catch (...) {
        Cerr << "Unhandled exception: " << CurrentExceptionMessage() << ". Process will terminate" << Endl;
        return 1;
    }
}

