#include <solomon/libs/cpp/clients/memstore/rpc.h>
#include <solomon/libs/cpp/host_resolver/host_resolver.h>
#include <solomon/libs/cpp/proto_convert/metric_type.h>
#include <solomon/libs/cpp/stockpile_codec/metric_archive.h>
#include <solomon/libs/cpp/string_pool/string_pool.h>
#include <solomon/libs/cpp/ts_model/visit.h>

#include <library/cpp/colorizer/colors.h>
#include <library/cpp/getopt/small/completer.h>
#include <library/cpp/getopt/small/last_getopt.h>
#include <library/cpp/getopt/small/modchooser.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/stream/file.h>

using namespace NSolomon;
using namespace NSolomon::NMemStore;
using namespace yandex::monitoring::memstore;

class TCommandBase: public TMainClassArgs {
protected:
    void RegisterOptions(NLastGetopt::TOpts& opts) override {
        TMainClassArgs::RegisterOptions(opts);

        opts.SetFreeArgsNum(0);
        opts.AddLongOption("host")
            .RequiredArgument("<host>")
            .StoreResult(&Host_)
            .DefaultValue("localhost:4780")
            .Help("hostname or conductor group of a memstore instance")
            .CompletionArgHelp("memstore host:port or conductor_group://group:port")
            .Completer(NLastGetopt::NComp::Host());
        opts.AddLongOption('v', "verbose")
            .StoreTrue(&Verbose_)
            .Help("print logs");
    }

    TAddressSet ResolveAddress() {
        TAddressSet res;
        if (Verbose_) Cerr << NColorizer::ST_DARK << "Resolving hosts..." << NColorizer::RESET << Endl;
        ConductorGroupResolver()->Resolve(Host_, &res);
        if (Verbose_) Cerr << NColorizer::ST_DARK << "Found " << res.size() << " host(s)" << NColorizer::RESET << Endl;
        return res;
    }

    TString ResolveNumId(ui32 numId) {
        auto hosts = ResolveAddress();

        if (hosts.size() == 1) {
            return *hosts.begin();
        }

        if (Verbose_) Cerr << NColorizer::ST_DARK << "Resolving num id..." << NColorizer::RESET << Endl;

        for (auto& host: hosts) {
            if (Verbose_) Cerr << NColorizer::ST_DARK << "Checking " << host << NColorizer::RESET << Endl;

            auto client = Client(host);

            ListShardsRequest req;
            auto shards = client->ListShards(req).ExtractValueSync();
            if (shards.Fail()) {
                PrintError(shards.Error());
            } else {
                auto& numIds = shards.Value().num_ids();
                if (std::find(numIds.begin(), numIds.end(), numId) != numIds.end()) {
                    Cerr << "Using host " << host << NColorizer::RESET << Endl;
                    return host;
                }
            }
        }

        Cerr << NColorizer::RED << "Unable to find shard " << numId << NColorizer::RESET << Endl;
        std::exit(1);
    }

    static void PrintError(const NGrpc::TGrpcStatus& status) {
        Cerr << NColorizer::RED << "Error code " << status.GRpcStatusCode << ": " << NColorizer::RESET;
        Cerr << status.Msg << Endl;
    }

    std::shared_ptr<IMemStoreRpc> Client(TString host) {
        yandex::solomon::config::rpc::TGrpcClientConfig conf;
        *conf.add_addresses() = std::move(host);
        return CreateNodeGrpc(conf, *NMonitoring::TMetricRegistry::Instance());
    }

protected:
    TString Host_;
    bool Verbose_ = false;
};

class TListShards: public TCommandBase {
protected:

    void RegisterOptions(NLastGetopt::TOpts& opts) override {
        TCommandBase::RegisterOptions(opts);

        opts.SetTitle("list currently know shards");

        opts.AddLongOption("generation")
            .RequiredArgument("<generation>")
            .StoreResult(&Generation_)
            .DefaultValue("0")
            .Completer(NLastGetopt::NComp::Host());
    }

    int DoRun(NLastGetopt::TOptsParseResult&&) override {
        ListShardsRequest req;
        req.set_generation(Generation_);

        auto addrs = ResolveAddress();
        if (addrs.empty()) {
            Cerr << NColorizer::RED << "No hosts in this conductor group" << NColorizer::RESET << Endl;
            return 1;
        }

        int err = 0;

        for (auto& addr: addrs) {
            Cerr << addr << Endl;

            auto res = Client(addr)->ListShards(req).ExtractValueSync();

            if (res.Success()) {
                auto& val = res.Value();
                Cerr << NColorizer::GREEN << "    Ok" << NColorizer::RESET << Endl;
                Cerr << NColorizer::ST_DARK << "    generation: " << NColorizer::RESET << val.generation() << Endl;
                if (Generation_ == val.generation()) {
                    Cerr << "    shards has not changed since the last request";
                } else if (val.num_ids_size() == 0) {
                    Cerr << "    no shards found" << Endl;
                } else {
                    Cerr << NColorizer::ST_DARK << "    shards (" << val.num_ids_size() << "):" << NColorizer::RESET << Endl;
                    int size = val.num_ids_size();
                    if (size != val.shard_keys_size()) {
                        Cerr << "invalid response num_ids_size(" << size << ") != shard_keys_size(" << val.shard_keys_size() << ')' << Endl;
                        return 1;
                    }

                    for (int i = 0; i < size; i++) {
                        ui32 numId = val.num_ids(i);
                        const auto& shardKey = val.shard_keys(i);
                        Cerr << addr << '\t' << numId << '\t' << shardKey.project() << ':' << shardKey.cluster() << ':' << shardKey.service() << Endl;
                    }
                }
            } else {
                PrintError(res.Error());
                err = 1;
            }
        }

        return err;
    }

protected:
    ui64 Generation_ = 0;
};

class TWrite: public TCommandBase {
protected:
    void RegisterOptions(NLastGetopt::TOpts& opts) override {
        TCommandBase::RegisterOptions(opts);

        opts.SetTitle("list currently know shards");

        opts.AddLongOption("meta")
            .RequiredArgument("<file>")
            .Help("path to a file with metadata in slog format")
            .Required()
            .CompletionArgHelp("metadata file")
            .StoreResult(&Meta_)
            .Completer(NLastGetopt::NComp::File());

        opts.AddLongOption("data")
            .RequiredArgument("<file>")
            .Help("path to a file with data in slog format")
            .Required()
            .CompletionArgHelp("data file")
            .StoreResult(&Data_)
            .Completer(NLastGetopt::NComp::File());

        opts.AddLongOption('n', "num-id")
            .RequiredArgument("<num-id>")
            .Help("numeric identifier of the shard that will receive the data")
            .Required()
            .CompletionArgHelp("shard num-id (number)")
            .StoreResult(&NumId_);
    }

    int DoRun(NLastGetopt::TOptsParseResult&&) override {
        WriteRequest req;
        req.set_metadata(TFileInput{Meta_}.ReadAll());
        req.set_data(TFileInput{Data_}.ReadAll());
        req.set_num_id(NumId_);

        auto res = Client(ResolveNumId(NumId_))->Write(req).ExtractValueSync();

        if (res.Success()) {
            Cerr << NColorizer::GREEN << "Ok" << NColorizer::RESET << Endl;
            return 0;
        } else {
            PrintError(res.Error());
            return 1;
        }
    }

protected:
    TString Meta_;
    TString Data_;
    ui32 NumId_ = 0;
};

class TReadBase: public TCommandBase {
protected:
    void RegisterOptions(NLastGetopt::TOpts& opts) override {
        TCommandBase::RegisterOptions(opts);

        opts.AddLongOption('n', "num-id")
            .RequiredArgument("<num-id>")
            .Help("numeric identifier of the shard that will receive the data")
            .Required()
            .CompletionArgHelp("shard num-id (number)")
            .StoreResult(&NumId_);

        opts.AddLongOption('s', "selectors")
            .RequiredArgument("<selectors>")
            .Help("selectors to filter output")
            .CompletionArgHelp("selectors")
            .StoreResult(&Selectors_);

        opts.AddLongOption('d', "downsampling")
            .RequiredArgument("<grid_millis>")
            .Help("grid size for downsampling in millisec")
            .CompletionArgHelp("downsampling")
            .StoreResult(&GridSizeMillis_);
    }

protected:
    ui32 NumId_ = 0;
    TString Selectors_ = "";
    i32 GridSizeMillis_ = -1;
};

class TFind: public TReadBase {
protected:
    void RegisterOptions(NLastGetopt::TOpts& opts) override {
        TReadBase::RegisterOptions(opts);

        opts.SetTitle("find all metrics that correspond to the given selectors");

        opts.AddLongOption('l', "limit")
            .RequiredArgument("<limit>")
            .StoreResult(&Limit_)
            .DefaultValue("0")
            .Help("limit number of results to this value")
            .CompletionArgHelp("limit (number)");
    }

    int DoRun(NLastGetopt::TOptsParseResult&&) override {
        FindRequest req;
        req.set_num_id(NumId_);
        req.set_selectors(Selectors_);
        req.set_limit(Limit_);

        auto res = Client(ResolveNumId(NumId_))->Find(req).ExtractValueSync();

        if (res.Success()) {
            auto& val = res.Value();
            Cerr << NColorizer::GREEN << "Ok" << NColorizer::RESET << Endl;
            Cerr << NColorizer::ST_DARK << "total count: " << NColorizer::RESET << val.total_count() << Endl;
            Cerr << NColorizer::ST_DARK << "truncated: " << NColorizer::RESET << val.truncated() << Endl;
            if (val.metrics_size() == 0) {
                Cerr << "no metrics found" << Endl;
            } else {
                NStringPool::TStringPool pool{val.string_pool()};
                Cerr << NColorizer::ST_DARK << "metrics (" << val.metrics_size() << "):" << NColorizer::RESET << Endl;
                for (auto& metric: val.metrics()) {
                    Y_VERIFY(metric.labels_idx_size() % 2 == 0);
                    Cerr << "    " << yandex::solomon::model::MetricType_Name(metric.type()) << " {";
                    TStringBuf sep = "";
                    for (auto it = metric.labels_idx().begin(); it != metric.labels_idx().end(); ) {
                        Cerr << sep;
                        Cerr << TString{pool[*it++]}.Quote();
                        Cerr << "=";
                        Cerr << TString{pool[*it++]}.Quote();
                        sep = ", ";
                    }
                    Cerr << "}";
                    Cerr << Endl;
                }
            }

            return 0;
        } else {
            PrintError(res.Error());
            return 1;
        }
    }

protected:
    size_t Limit_ = 0;
};

class TRead: public TReadBase {
protected:
    void RegisterOptions(NLastGetopt::TOpts& opts) override {
        TReadBase::RegisterOptions(opts);

        opts.SetTitle("read all metrics that correspond to the given selectors");
    }

    int DoRun(NLastGetopt::TOptsParseResult&&) override {
        ReadManyRequest req;
        req.set_num_id(NumId_);
        req.mutable_lookup()->set_selectors(Selectors_);
        if (GridSizeMillis_ > 0) {
            auto op = req.add_operations();
            yandex::solomon::math::Operation downsampling;
            downsampling.mutable_downsampling()->set_grid_millis(GridSizeMillis_);
            *op = downsampling;
        }

        auto res = Client(ResolveNumId(NumId_))->ReadMany(req).ExtractValueSync();

        if (res.Success()) {
            auto& val = res.Value();
            Cerr << NColorizer::GREEN << "Ok" << NColorizer::RESET << Endl;
            if (val.metrics_size() == 0) {
                Cerr << "no metrics found" << Endl;
            } else {
                NStringPool::TStringPool pool{val.string_pool()};
                Cerr << NColorizer::ST_DARK << "metrics (" << val.metrics_size() << "):" << NColorizer::RESET << Endl;
                for (auto& metric: val.metrics()) {
                    Y_VERIFY(metric.labels_idx_size() % 2 == 0);
                    Cerr << "    " << yandex::solomon::model::MetricType_Name(metric.type()) << " {";
                    TStringBuf sep = "";
                    for (auto it = metric.labels_idx().begin(); it != metric.labels_idx().end(); ) {
                        Cerr << sep;
                        Cerr << TString{pool[*it++]}.Quote();
                        Cerr << "=";
                        Cerr << TString{pool[*it++]}.Quote();
                        sep = ", ";
                    }
                    Cerr << "}";
                    Cerr << Endl;
                    for (auto& chunk: metric.time_series().chunks()) {
                        NStockpile::TMetricArchiveCodec codec{
                            static_cast<NStockpile::EFormat>(metric.time_series().format_version())};
                        auto input = NStockpile::TCodecInput{chunk.content()};
                        auto data = codec.Decode(&input);

                        NTsModel::Visit(NSolomon::NTsModel::FromProto(data.Header().Type).Value(), [&](auto traits) {
                            using TPoint = typename decltype(traits)::TPoint;
                            using TDecoder = NTsModel::TDecoder<TPoint>;

                            auto point = traits.MakePoint();
                            auto decoder = TDecoder{data.Columns(), data.Data()};
                            while (decoder.NextPoint(&point)) {
                                Cerr << "        " << point << Endl;
                            }
                        });
                    }
                }
            }

            return 0;
        } else {
            PrintError(res.Error());
            return 1;
        }
    }
};

class TMain: public TMainClassModes {
protected:
    void RegisterModes(TModChooser& modes) override {
        TMainClassModes::RegisterModes(modes);

        modes.SetDescription("memstore client for humans (robots shall use grpc_cli)");

        modes.AddGroupModeDescription("observe");

        modes.AddMode("list-shards", new TListShards{}, "list currently know shards");
        modes.AddAlias("ls", "list-shards");

        modes.AddGroupModeDescription("write");

        modes.AddMode("write", new TWrite{}, "write data to memstore");

        modes.AddGroupModeDescription("read");

        modes.AddMode("find", new TFind{}, "find all metrics that correspond to the given selectors");
        modes.AddMode("read", new TRead{}, "read all metrics that correspond to the given selectors");

        modes.AddGroupModeDescription("auxiliary commands");

        modes.AddCompletions("memstore-cli");
    }
};

int main(int argc, const char** argv) {
    return TMain{}.Run(argc, argv);
}
