#include "rpc.h"

#include <solomon/services/memstore/api/memstore_service.grpc.pb.h>

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/random/random.h>
#include <util/generic/hash_set.h>

namespace NSolomon::NMemStore {

using yandex::solomon::config::rpc::TGrpcClientConfig;
using namespace yandex::monitoring::memstore;

namespace {

class TMemStoreRpcImpl final: public IMemStoreRpc {
public:
    TMemStoreRpcImpl(
            NMonitoring::IMetricRegistry&,
            std::unique_ptr<TGrpcServiceConnection<MemStoreService>> connection)
        : Connection_{std::move(connection)}
    {
    }

    TWriteAsyncResponse Write(const WriteRequest& request) override {
        return Request<WriteRequest, WriteResponse, &MemStoreService::Stub::AsyncWrite>(request);
    }

    TListShardsAsyncResponse ListShards(const ListShardsRequest& request) override {
        return Request<ListShardsRequest, ListShardsResponse, &MemStoreService::Stub::AsyncListShards>(request);
    }

    TFindAsyncResponse Find(const FindRequest& request) override {
        return Request<FindRequest, FindResponse, &MemStoreService::Stub::AsyncFind>(request);
    }

    TLabelKeysAsyncResponse LabelKeys(const LabelKeysRequest& request) override {
        return Request<LabelKeysRequest, LabelKeysResponse, &MemStoreService::Stub::AsyncLabelKeys>(request);
    }

    TLabelValuesAsyncResponse LabelValues(const LabelValuesRequest& request) override {
        return Request<LabelValuesRequest, LabelValuesResponse, &MemStoreService::Stub::AsyncLabelValues>(request);
    }

    TUniqueLabelsAsyncResponse UniqueLabels(const UniqueLabelsRequest& request) override {
        return Request<UniqueLabelsRequest, UniqueLabelsResponse, &MemStoreService::Stub::AsyncUniqueLabels>(request);
    }

    TReadOneAsyncResponse ReadOne(const ReadOneRequest& request) override {
        return Request<ReadOneRequest, ReadOneResponse, &MemStoreService::Stub::AsyncReadOne>(request);
    }

    TReadManyAsyncResponse ReadMany(const ReadManyRequest& request) override {
        return Request<ReadManyRequest, ReadManyResponse, &MemStoreService::Stub::AsyncReadMany>(request);
    }

    void Stop(bool) override {
        // nop
    }

private:
    template <typename TReq, typename TRes, auto Request>
    NThreading::TFuture<TErrorOr<TRes, NGrpc::TGrpcStatus>> Request(const TReq& request) {
        auto promise = NThreading::NewPromise<TErrorOr<TRes, NGrpc::TGrpcStatus>>();
        auto cb = [promise] (NGrpc::TGrpcStatus&& status, TRes&& result) mutable {
            if (!status.Ok()) {
                promise.SetValue(std::move(status));
            } else {
                promise.SetValue(std::move(result));
            }
        };

        Connection_->Request<TReq, TRes>(
            request,
            std::move(cb),
            Request);

        return promise.GetFuture();
    }

private:
    std::unique_ptr<TGrpcServiceConnection<MemStoreService>> Connection_;
};

std::unique_ptr<IMemStoreRpc> CreateMemstoreClient(
        NMonitoring::IMetricRegistry& registry,
        std::unique_ptr<TGrpcServiceConnection<MemStoreService>> connection)
{
    return std::make_unique<TMemStoreRpcImpl>(registry, std::move(connection));
}

} // namespace

std::unique_ptr<IMemStoreRpc> CreateNodeGrpc(
    const yandex::solomon::config::rpc::TGrpcClientConfig& conf,
    NMonitoring::IMetricRegistry& registry,
    TString clientId)
{
    auto threadPool = CreateGrpcThreadPool(conf);
    auto sc = CreateGrpcServiceConnection<MemStoreService>(
        conf,
        false,
        registry,
        std::move(threadPool),
        std::move(clientId));

    return CreateMemstoreClient(registry, std::move(sc));
}

std::shared_ptr<IClusterRpc<IMemStoreRpc>> CreateClusterGrpc(
    const yandex::solomon::config::rpc::TGrpcClientConfig& conf,
    NMonitoring::IMetricRegistry& registry,
    TString clientId)
{
    return std::make_shared<TGrpcClusterClientBase<MemStoreService, IMemStoreRpc>>(
        conf,
        false,
        registry,
        CreateMemstoreClient,
        std::move(clientId));
}

} // namespace NSolomon::NMemStore
