#pragma once

#include "rpc.h"

#include <util/random/random.h>

namespace NSolomon::NMemStore {

/**
 * Stub implementation of MemStore RPC.
 */
class TMemStoreRpcStub: public IMemStoreRpc {
public:
    TWriteAsyncResponse Write(const yandex::monitoring::memstore::WriteRequest&) override {
        return NotImplemented<TWriteAsyncResponse>("Write");
    }

    TListShardsAsyncResponse ListShards(const yandex::monitoring::memstore::ListShardsRequest&) override {
        return NotImplemented<TListShardsAsyncResponse>("ListShards");
    }

    TFindAsyncResponse Find(const yandex::monitoring::memstore::FindRequest&) override {
        return NotImplemented<TFindAsyncResponse>("Find");
    }

    TLabelKeysAsyncResponse LabelKeys(const yandex::monitoring::memstore::LabelKeysRequest&) override {
        return NotImplemented<TLabelKeysAsyncResponse>("LabelKeys");
    }

    TLabelValuesAsyncResponse LabelValues(const yandex::monitoring::memstore::LabelValuesRequest&) override {
        return NotImplemented<TLabelValuesAsyncResponse>("LabelValues");
    }

    TUniqueLabelsAsyncResponse UniqueLabels(const yandex::monitoring::memstore::UniqueLabelsRequest&) override {
        return NotImplemented<TUniqueLabelsAsyncResponse>("UniqueLabels");
    }

    TReadOneAsyncResponse ReadOne(const yandex::monitoring::memstore::ReadOneRequest&) override {
        return NotImplemented<TReadOneAsyncResponse>("ReadOne");
    }

    TReadManyAsyncResponse ReadMany(const yandex::monitoring::memstore::ReadManyRequest&) override {
        return NotImplemented<TReadManyAsyncResponse>("ReadMany");
    }

    void Stop(bool /*wait*/) override {
        // nop
    }

protected:
    template <typename T>
    static T NotImplemented(TStringBuf method) {
        return NThreading::MakeErrorFuture<NThreading::TFutureType<T>>(
            std::make_exception_ptr(yexception() << "MemStoreRpc::" << method << "() is not implemented"));
    }

    template <typename T>
    static T GrpcError(grpc::StatusCode statusCode, TString message) {
        using U = NThreading::TFutureType<T>;
        return NThreading::MakeFuture<U>(U::FromError(NGrpc::TGrpcStatus(statusCode, std::move(message))));
    }

    template <typename T>
    static T Fail(TStringBuf message) {
        using U = NThreading::TFutureType<T>;
        return NThreading::MakeErrorFuture<U>(std::make_exception_ptr(yexception() << message));
    }
};

class TMemStoreClusterRpcStub: public IMemStoreClusterRpc {
public:
    explicit TMemStoreClusterRpcStub(THashMap<TString, std::unique_ptr<IMemStoreRpc>> nodeRpc)
        : NodeRpc_{std::move(nodeRpc)}
    {
        Addresses_.reserve(NodeRpc_.size());
        for (const auto& [k, v]: NodeRpc_) {
            Addresses_.emplace_back(k);
        }
    }

    IMemStoreRpc* Get(TStringBuf address) noexcept override {
        auto it = NodeRpc_.find(address);
        return it == NodeRpc_.end() ? nullptr : it->second.get();
    }

    IMemStoreRpc* GetAny() noexcept override {
        auto idx = RandomNumber<size_t>(Addresses_.size());
        return Get(Addresses_[idx]);
    }

    TString GetAnyAddress() const noexcept override {
        auto idx = RandomNumber<size_t>(Addresses_.size());
        return Addresses_[idx];
    }

    const std::vector<TString>& Addresses() const noexcept override {
        return Addresses_;
    }

    void Add(TStringBuf) override {
        ythrow yexception() << "not implemented";
    }

    void Stop(bool /*wait*/) override {
        // nop
    }

private:
    THashMap<TString, std::unique_ptr<IMemStoreRpc>> NodeRpc_;
    std::vector<TString> Addresses_;
};


} // namespace NSolomon::NMemStore
