#pragma once

#include <util/digest/multi.h>
#include <util/generic/hash.h>
#include <util/generic/string.h>
#include <util/generic/vector.h>
#include <util/stream/output.h>
#include <util/system/types.h>

#include <concepts>
#include <iostream>
#include <set>

namespace NSolomon::NSlicer::NApi {

// TODO(ivanzhukov@): declare a common TNumId type for all components
using TNumId = ui32;
/**
 * In the Slicer paper, slice key is a 63-bit hash that maps an application-defined key to a uniform distribution.
 * But NumId is already uniformly distributed, so we will use it as a slice key without hashing.
 * Thus, in our implementation, slice key is 32-bit
 */
struct TSlice {
    TNumId Start{0};
    /**
     * inclusive
     */
    TNumId End{0};

    TSlice() = default;

    TSlice(TNumId start, TNumId end)
        : Start{start}
        , End{end}
    {
        Y_VERIFY(Start <= End, "slice's Start cannot be > End");
    }

    /**
     * Number of num ids inside a slice
     */
    ui32 Size() const {
        return End - Start + 1;
    }

    bool operator<(const TSlice& other) const {
        return End < other.Start;
    }

    bool operator>(const TSlice& other) const {
        return Start > other.End;
    }

    bool operator==(const TSlice& other) const = default;
};

// TODO(ivanzhukov): add tests
struct TSliceTransparentComparator {
    using is_transparent = void;

    bool operator()(const TSlice& lhs, const TSlice& rhs) const {
        return lhs < rhs;
    }

    /**
     * used in upper_bound() calls to find a slice _containing_ a given numId
     */
    bool operator()(const TNumId& numId, const TSlice& rhs) const {
        return numId <= rhs.End;
    }

    bool operator()(const TSlice& lhs, const TNumId& numId) const {
        return lhs.End < numId;
    }
};

using TSlices = std::set<TSlice, TSliceTransparentComparator>;
// TODO(ivanzhukov, SOLOMON-7762): make it a separate structure and use TNodeEndpoint ("fqdn:port") instead of TString
using THosts = TVector<TString>;

IOutputStream& operator<<(IOutputStream& os, const NSolomon::NSlicer::NApi::TSlice& slice);
std::ostream& operator<<(std::ostream& os, const NSolomon::NSlicer::NApi::TSlice& slice);

IOutputStream& operator<<(IOutputStream& os, const NSolomon::NSlicer::NApi::TSlices& slices);
std::ostream& operator<<(std::ostream& os, const NSolomon::NSlicer::NApi::TSlices& slices);

IOutputStream& operator<<(IOutputStream& os, const NSolomon::NSlicer::NApi::THosts& hosts);
std::ostream& operator<<(std::ostream& os, const NSolomon::NSlicer::NApi::THosts& hosts);

} // namespace NSolomon::NSlicer::NApi

template <>
struct THash<NSolomon::NSlicer::NApi::TSlice> {
    ui64 operator()(const NSolomon::NSlicer::NApi::TSlice& slice) const noexcept {
        return MultiHash(slice.Start, slice.End);
    }
};
