#include <solomon/libs/cpp/clients/slicer/slicer_client.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

using namespace NSolomon::NSlicerClient;

NMonitoring::TMetricRegistry DUMMY_REGISTRY;

class TCommandBase {
public:
    TCommandBase()
        : Opts_{NLastGetopt::TOpts::Default()}
    {
        Opts_.AddLongOption('p', "port", "MemStore port to connect to")
            .RequiredArgument("PORT")
            .StoreResult(&Port_)
            .DefaultValue(4710);
        Opts_.AddLongOption('h', "host", "MemStore host to connect to")
            .RequiredArgument("HOST")
            .StoreResult(&Host_)
            .DefaultValue("localhost");
    }

    void Execute(int argc, char** argv) {
        Args_.Reset(new NLastGetopt::TOptsParseResult(&Opts_, argc, argv));

        yandex::solomon::config::rpc::TGrpcClientConfig conf;
        auto* addr = conf.AddAddresses();
        *addr = TStringBuilder() << Host_ << ':' << Port_;
        DoExecute(
                CreateSlicerGrpcClient(std::move(conf), DUMMY_REGISTRY)
        );
    }

    virtual ~TCommandBase() = default;

    virtual TStringBuf Name() const = 0;
    virtual TStringBuf Help() const = 0;

protected:
    virtual void DoExecute(ISlicerClientPtr slicerClient) = 0;

protected:
    NLastGetopt::TOpts Opts_;
    THolder<NLastGetopt::TOptsParseResult> Args_;
    TString Host_;
    ui16 Port_;
};

class TGetAllAssignmentsCommand: public TCommandBase {
public:
    TGetAllAssignmentsCommand() {
        Opts_.AddLongOption("service", "service within which a request is being made")
            .RequiredArgument("STRING");
    }

private:
    TStringBuf Name() const override {
        return Name_;
    }

    void DoExecute(ISlicerClientPtr slicerClient) override {
        auto service = GetService();
        auto assignmentsOrErrPtr = slicerClient->GetAllAssignments(TString{service}).ExtractValueSync();
        auto& assignmentsOrErr = *assignmentsOrErrPtr;

        if (!assignmentsOrErr.Success()) {
            Cerr << "failed to get all assignments (code: " << assignmentsOrErr.Error().GRpcStatusCode << "): "
                 << assignmentsOrErr.Error().Msg << Endl;
            exit(1);
        }

        const auto& value = assignmentsOrErr.Value();

        for (auto&& assignment: value.assignments()) {
            Cout << "[" << assignment.slice_start() << "; " << assignment.slice_end() << "]: ";

            const auto& hosts = assignment.hosts();

            for (int i = 0; i != hosts.size(); ++i) {
                Cout << hosts[i];

                if (i < hosts.size() - 1) {
                    Cout << ", ";
                }
            }

            Cout << Endl;
        }

        Cout << Endl;
        Cout << "is_leader: " << value.is_leader() << Endl;

        if (!value.is_leader()) {
            Cout << "leader_address: " << value.leader_address() << Endl;
        }
    }

    TStringBuf GetService() const {
        TStringBuf key = Args_->Get("service");
        Y_ENSURE(!key.empty(), "service value cannot be empty");

        return key;
    }

    TStringBuf Help() const override {
        return Help_;
    }

private:
    const TString Name_{"all-assignments"};
    const TString Help_{"Dumps all assignments"};
};

class TGetSlicesByHostCommand: public TCommandBase {
public:
    TGetSlicesByHostCommand() {
        Opts_.AddLongOption("service", "service within which a request is being made")
            .RequiredArgument("STRING");
        Opts_.AddLongOption("host", "host (node fqdn) for which slices are requested")
                .RequiredArgument("STRING");
    }

private:
    TStringBuf Name() const override {
        return Name_;
    }

    void DoExecute(ISlicerClientPtr slicerClient) override {
        auto host = GetHost();
        auto service = GetService();
        auto slicesOrErrorPtr = slicerClient->GetSlicesByHost(TString{service}, TString{host}).ExtractValueSync();
        auto& slicesOrError = *slicesOrErrorPtr;

        if (!slicesOrError.Success()) {
            Cerr << "failed to get slices for host " << host << " (code: " << slicesOrError.Error().GRpcStatusCode << "): "
                    << slicesOrError.Error().Msg << Endl;
            exit(1);
        }

        Cout << "assigned shards for host " << host << ": ";

        const auto& value = slicesOrError.Value();

        for (int i = 0; i != value.assigned_starts_size(); ++i) {
            Cout << "[" << value.assigned_starts(i) << "; " << value.assigned_ends(i) << "]";

            if (i < value.assigned_starts_size() - 1) {
                Cout << ", ";
            }
        }

        Cout << Endl;
        Cout << "unassigned shards for host " << host << ": ";

        for (int i = 0; i != value.unassigned_starts_size(); ++i) {
            Cout << "[" << value.unassigned_starts(i) << "; " << value.unassigned_ends(i) << "]";

            if (i < value.unassigned_starts_size() - 1) {
                Cout << ", ";
            }
        }

        Cout << Endl;
        Cout << Endl;
        Cout << "is_leader: " << value.is_leader() << Endl;

        if (!value.is_leader()) {
            Cout << "leader_address: " << value.leader_address() << Endl;
        }
    }

    TStringBuf Help() const override {
        return Help_;
    }

    TStringBuf GetHost() const {
        TStringBuf key = Args_->Get("host");
        Y_ENSURE(!key.empty(), "host value cannot be empty");

        return key;
    }

    TStringBuf GetService() const {
        TStringBuf key = Args_->Get("service");
        Y_ENSURE(!key.empty(), "service value cannot be empty");

        return key;
    }

private:
    const TString Name_{"assigned-shards"};
    const TString Help_{"Dumps shards assigned to a host"};
};

bool IsHelp(TStringBuf arg) {
    return arg == "-h" || arg == "--help" || arg == "help";
}

using TCommands = std::array<THolder<TCommandBase>, 2>;
void PrintUsage(TStringBuf executable, TCommands& commands) {
    Cerr << executable << " <command> [-h|--host] [-p|--port] [OPTIONS]\n\n";

    for (auto& command: commands) {
        Cerr << command->Name() << '\t' << command->Help() << '\n';
    }
}

int main(int argc, char** argv) {
    using namespace NLastGetopt;

    TCommands commands {
        MakeHolder<TGetAllAssignmentsCommand>(),
        MakeHolder<TGetSlicesByHostCommand>(),
    };

    if (argc <= 1 || IsHelp(argv[1])) {
        PrintUsage(argv[0], commands);
        return 0;
    }

    auto it = FindIf(commands.begin(), commands.end(), [arg = TStringBuf{argv[1]}] (auto&& cmd){
        return cmd->Name() == arg;
    });

    if (it == commands.end()) {
        Cerr << "Unknown command: " << argv[1] << Endl;
        PrintUsage(argv[0], commands);
        return 1;
    }

    (*it)->Execute(argc, argv);
}
