#include "slicelet_listener.h"

#include <solomon/libs/cpp/actors/util/requests_executor.h>
#include <solomon/libs/cpp/cluster_map/cluster.h>
#include <solomon/libs/cpp/logging/logging.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/events.h>
#include <library/cpp/actors/core/hfunc.h>

#include <util/generic/scope.h>

#include <utility>

namespace NSolomon::NSlicerClient {
namespace {

using namespace NActors;

class TSliceletPrivateEvents: TPrivateEvents {
    enum {
        UpdateAssignments = SpaceBegin,
        RequestToSlicerCompleted,
        End
    };

    static_assert(End < SpaceEnd, "too many event types");

public:
    struct TUpdateAssignments: TEventLocal<TUpdateAssignments, UpdateAssignments> {
    };

    struct TRequestToSlicerCompleted: TEventLocal<TRequestToSlicerCompleted, RequestToSlicerCompleted> {
        TGetSlicesByHostResponseOrErrorPtr Response;

        explicit TRequestToSlicerCompleted(TGetSlicesByHostResponseOrErrorPtr&& response)
            : Response{std::move(response)}
        {}
    };
};

class TRequestToSlicerCtx: public IRequestCtx {
public:
    TRequestToSlicerCtx(ISlicerClient* slicerClient, const TString& slicerEndpoint, TString service, TString host)
        : SlicerClient_{std::move(slicerClient)}
        , SlicerEndpoint_{slicerEndpoint}
        , Service_{std::move(service)}
        , Host_{std::move(host)}
    {
        What_ = TStringBuilder() << "[GetSlicesByHost() from " << slicerEndpoint << "]";
    }

private:
    NThreading::TFuture<void> PerformRequest() override {
        ++TimesPerformed_;

        auto request = SlicerClient_->GetSlicesByHost(Service_, Host_);
        return HandleResult(request);
    }

    NThreading::TFuture<void> HandleResult(TAsyncGetSlicesByHostResponse& result) {
        return result
                .Subscribe([this, reqCtxPtr{shared_from_this()}](TAsyncGetSlicesByHostResponse future) {
                    // TODO(ivanzhukov): collect/log all errors, not only the last one
                    try {
                        Result_ = std::move(future.ExtractValueSync());
                        /*
                         * TODO(ivanzhukov):
                         *  An extremely weird bug could happen here: if running with no backoff,
                         *  Result_->Fail results in std::variant<>::index which causes a SIGSEGV.
                         *  Should be examined more carefully
                         */
                        ShouldRetry_ = Result_->Fail();
                    } catch (...) {
                        Result_ = std::make_unique<TGetSlicesByHostResponseOrError>(
                                NGrpc::TGrpcStatus{CurrentExceptionMessage(), grpc::StatusCode::INTERNAL, true});

                        ShouldRetry_ = false;
                    }
                })
                .IgnoreResult();
    }

    bool ShouldRetry() const override {
        // TODO(ivanzhukov): take into account cases when it shouldn't be retried (exceptions, network overload, etc.)
        return ShouldRetry_;
    }

    TString What() const override {
        return What_;
    }

    size_t TimesPerformed() const override {
        return TimesPerformed_;
    }

    std::unique_ptr<IEventBase> MakeReplyEvent() override {
        return nullptr;
    }

    std::unique_ptr<IEventBase> MakeCompletionEvent() override {
        return std::make_unique<TSliceletPrivateEvents::TRequestToSlicerCompleted>(std::move(Result_));
    }

private:
    ISlicerClient* SlicerClient_;
    TString SlicerEndpoint_;
    TString Service_;
    TString Host_;

    TString What_;
    TGetSlicesByHostResponseOrErrorPtr Result_;
    size_t TimesPerformed_{0};
    bool ShouldRetry_{false};
};

class TSliceletListenerActor: public NActors::TActorBootstrapped<TSliceletListenerActor>, TPrivateEvents {
public:
    explicit TSliceletListenerActor(TSliceletListenerOptions opts)
        : Options_{std::move(opts)}
        , SlicerClients_(Options_.SlicerClients)
        , CurrentSlicerAddress_{}
    {
        Y_ENSURE(!Options_.Service.empty(), "service is not specified");
        Y_ENSURE(!Options_.Host.empty(), "host is not specified");
    }

// SL -- Slicelet Listener
#define SL_TRACE(...) MON_TRACE(SliceletListener, __VA_ARGS__)
#define SL_DEBUG(...) MON_DEBUG(SliceletListener, __VA_ARGS__)
#define SL_INFO(...)  MON_INFO(SliceletListener,  __VA_ARGS__)
#define SL_WARN(...)  MON_WARN(SliceletListener,  __VA_ARGS__)
#define SL_ERROR(...) MON_ERROR(SliceletListener, __VA_ARGS__)
#define SL_CRIT(...)  MON_CRIT(SliceletListener,  __VA_ARGS__)

    void Bootstrap() {
        if (SlicerClients_) {
            Become(&TThis::StateWork);

            CurrentSlicerAddress_ = SlicerClients_->GetAnyAddress();
            CurrentSlicer_ = SlicerClients_->Get(CurrentSlicerAddress_);

            SL_INFO("starting with a slicer: " << CurrentSlicerAddress_);
            ScheduleRequestToSlicer();
        } else {
            SL_INFO("no slicer clients -- switching to the idle state");
            Become(&TThis::Idle);
        }
    }

    STATEFN(StateWork) {
        switch (ev->GetTypeRewrite()) {
            cFunc(TEvents::TSystem::Poison, PassAway);
            hFunc(TSliceletPrivateEvents::TUpdateAssignments, OnUpdateAssignments);
            hFunc(TSliceletPrivateEvents::TRequestToSlicerCompleted, OnRequestToSlicerCompleted);
            hFunc(TSliceletListenerEvents::TEvSubscribe, OnSubscribe);
            hFunc(TSliceletListenerEvents::TEvUnsubscribe, OnUnsubscribe);
        }
    }

    STATEFN(Idle) {
        switch (ev->GetTypeRewrite()) {
            cFunc(TEvents::TSystem::Poison, PassAway);
        }
    }

private:
    void ScheduleRequestToSlicer() {
        // TODO: add a jitter
        auto overcommit = TInstant::Now().MilliSeconds() % Options_.UpdateInterval.MilliSeconds();
        auto wait = TDuration::MilliSeconds(Options_.UpdateInterval.MilliSeconds() - overcommit);

        Schedule(wait, new TSliceletPrivateEvents::TUpdateAssignments);
    }

    void OnUpdateAssignments(const TSliceletPrivateEvents::TUpdateAssignments::TPtr&) {
        TVector<IRequestCtxPtr> requests;
        requests.emplace_back(std::make_shared<TRequestToSlicerCtx>(
                CurrentSlicer_,
                CurrentSlicerAddress_,
                Options_.Service,
                Options_.Host
        ));

        TRequestsExecutorOptions opts;
        opts.SetMaxRetries(Options_.MaxRetries)
                .SetBackoff(Options_.RetriesBackoff)
                ;
        RegisterWithSameMailbox(CreateRequestsExecutorActor(opts, requests, SelfId()).Release());
    }

    void OnSubscribe(const TSliceletListenerEvents::TEvSubscribe::TPtr& ev) {
        Subscribers_.emplace(ev->Sender);
    }

    void OnUnsubscribe(const TSliceletListenerEvents::TEvUnsubscribe::TPtr& ev) {
        Subscribers_.erase(ev->Sender);
    }

    void OnRequestToSlicerCompleted(const TSliceletPrivateEvents::TRequestToSlicerCompleted::TPtr& ev) {
        auto respOrErr = std::move(ev->Get()->Response);

        if (respOrErr->Fail()) {
            auto prevAddress = CurrentSlicerAddress_;
            const auto& err = respOrErr->Error();

            CurrentSlicerAddress_ = SlicerClients_->GetAnyAddress();
            CurrentSlicer_ = SlicerClients_->Get(CurrentSlicerAddress_);

            if (err.InternalError) {
                SL_ERROR("internal error while gathering info from " << prevAddress << ": " << err.Msg);
                SL_ERROR("switching to: " << CurrentSlicerAddress_);
            } else {
                SL_WARN("failed to get a response from " << prevAddress << ": (" << err.GRpcStatusCode << ") " << err.Msg);
                SL_WARN("switching to: " << CurrentSlicerAddress_);
            }

            ScheduleRequestToSlicer();
            return;
        }

        auto& resp = respOrErr->Value();

        if (!resp.is_leader()) {
            SL_DEBUG(CurrentSlicerAddress_ << " is not the leader");

            // TODO(ivanzhukov): the slicer follower should forward a req to a leader and respond with the answer
            if (auto leaderAddress = resp.leader_address()) {
                SL_DEBUG("got the leader address from the response: " << leaderAddress << ". Trying again");

                CurrentSlicerAddress_ = leaderAddress;
                CurrentSlicer_ = SlicerClients_->Get(leaderAddress);
            } else {
                CurrentSlicerAddress_ = SlicerClients_->GetAnyAddress();
                CurrentSlicer_ = SlicerClients_->Get(CurrentSlicerAddress_);

                SL_DEBUG("no leader address in the response. Switching to: " << CurrentSlicerAddress_);
            }

            ScheduleRequestToSlicer();
            return;
        }

        ScheduleRequestToSlicer();
        SL_TRACE("got the response from " << CurrentSlicerAddress_);

        // TODO(ivanzhukov): prevent a response from copying
        for (const auto& sub: Subscribers_) {
            Send(sub, new TSliceletListenerEvents::TSlicesUpdate{resp});
        }
    }

private:
    TSliceletListenerOptions Options_;
    ISlicerClusterClientPtr SlicerClients_;
    TString CurrentSlicerAddress_;
    ISlicerClient* CurrentSlicer_;
    THashSet<TActorId> Subscribers_;
};

class TSliceletListenerActorForTesting: public TActorBootstrapped<TSliceletListenerActorForTesting> {
public:
    explicit TSliceletListenerActorForTesting(NSlicer::NApi::TSlices slices)
        : Slices_{std::move(slices)}
    {
    }

    void Bootstrap() {
        Become(&TThis::Main);
    }

    STATEFN(Main) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TSliceletListenerEvents::TEvSubscribe, OnSubscribe);
        }
    }

private:
    void OnSubscribe(const TSliceletListenerEvents::TEvSubscribe::TPtr& evPtr) {
        yandex::monitoring::slicer::GetSlicesByHostResponse resp;

        for (const auto& slice: Slices_) {
            resp.add_assigned_starts(slice.Start);
            resp.add_assigned_ends(slice.End);

        }

        Send(evPtr->Sender, new TSliceletListenerEvents::TSlicesUpdate{std::move(resp)});
    }

private:
    NSlicer::NApi::TSlices Slices_;
};

void ResolveAddressesInGrpcClientConfig(yandex::solomon::config::rpc::TGrpcClientConfig& conf) {
    TStringBuilder sb;
    for (const auto& addr: conf.GetAddresses()) {
        sb << addr << '\n';
    }

    auto cluster = TClusterMapBase::Load(sb);
    conf.ClearAddresses();

    for (auto&& node: cluster->Nodes()) {
        conf.AddAddresses(node.Endpoint);
    }
}

} // namespace

NSlicerClient::TSliceletListenerOptions ConstructSliceletListenerOptions(
    yandex::solomon::config::slicer::SliceletConfig config,
    NMonitoring::TMetricRegistry& registry,
    const TString& localAddress)
{
    auto* grpcConfig = config.mutable_slicer_client_config();

    ResolveAddressesInGrpcClientConfig(*grpcConfig);

    auto slicerClients = NSlicerClient::CreateSlicerGrpcClusterClient(
        *grpcConfig,
        registry,
        config.client_id());

    return NSlicerClient::TSliceletListenerOptions{
        .Service = config.service(),
        .Host = localAddress,
        .UpdateInterval = FromProtoTime(config.update_period(), TDuration::Seconds(2)),
        .RetriesBackoff = FromProtoTime(config.retries_backoff(), TDuration::Seconds(2)),
        .MaxRetries = config.max_retries(),
        .SlicerClients = std::move(slicerClients),
    };
}

NActors::IActor* CreateSliceletListenerActor(TSliceletListenerOptions opts) {
    return new TSliceletListenerActor{std::move(opts)};
}

NActors::IActor* CreateSliceletListenerActorForTesting(NSlicer::NApi::TSlices slices) {
    return new TSliceletListenerActorForTesting{std::move(slices)};
}

} // namespace NSolomon::NSlicerClient
