#include "slicer_client.h"

#include <solomon/services/slicer/api/slicer_service.grpc.pb.h>

#include <util/random/entropy.h>
#include <util/random/fast.h>

namespace NSolomon::NSlicerClient {
namespace {

using namespace yandex::monitoring::slicer;

using NGrpc::TGrpcStatus;
using NThreading::NewPromise;
using yandex::solomon::config::rpc::TGrpcClientConfig;
using namespace NMonitoring;

class TGrpcSlicerClientBase: public ISlicerClient {
public:
    TGrpcSlicerClientBase(
            IMetricRegistry&,
            std::unique_ptr<TGrpcServiceConnection<SlicerService>> connection)
        : Connection_{std::move(connection)}
    {
    }

    TAsyncGetAllAssignmentsResponse GetAllAssignments(TString service) noexcept override {
        GetAllAssignmentsRequest req;
        req.set_service(std::move(service));

        auto promise = NewPromise<TGetAllAssignmentsResponseOrErrorPtr>();

        auto cb = [promise] (TGrpcStatus&& status, GetAllAssignmentsResponse&& result) mutable {
            if (!status.Ok()) {
                promise.SetValue(std::make_unique<TGetAllAssignmentsResponseOrError>(std::move(status)));
            } else {
                promise.SetValue(std::make_unique<TGetAllAssignmentsResponseOrError>(std::move(result)));
            }
        };

        Connection_->Request<GetAllAssignmentsRequest, GetAllAssignmentsResponse>(
            req,
            std::move(cb),
            &SlicerService::Stub::AsyncGetAllAssignments);

        return promise.GetFuture();
    }

    TAsyncGetSlicesByHostResponse GetSlicesByHost(TString service, TString nodeFqdn) noexcept override {
        GetSlicesByHostRequest req;
        req.set_host(std::move(nodeFqdn));
        req.set_service(std::move(service));

        auto promise = NewPromise<TGetSlicesByHostResponseOrErrorPtr>();

        auto cb = [promise] (TGrpcStatus&& status, GetSlicesByHostResponse&& result) mutable {
            if (!status.Ok()) {
                promise.SetValue(std::make_unique<TGetSlicesByHostResponseOrError>(std::move(status)));
            } else {
                promise.SetValue(std::make_unique<TGetSlicesByHostResponseOrError>(std::move(result)));
            }
        };

        Connection_->Request<GetSlicesByHostRequest, GetSlicesByHostResponse>(
                req,
                std::move(cb),
                &SlicerService::Stub::AsyncGetSlicesByHost);

        return promise.GetFuture();
    }

    void Stop(bool) noexcept override {
    }

private:
    std::unique_ptr<TGrpcServiceConnection<SlicerService>> Connection_;
};

std::unique_ptr<ISlicerClient> CreateSlicerClient(
    IMetricRegistry& registry,
    std::unique_ptr<TGrpcServiceConnection<SlicerService>> connection)
{
    return std::make_unique<TGrpcSlicerClientBase>(registry, std::move(connection));
}

} // namespace

std::unique_ptr<ISlicerClient> CreateSlicerGrpcClient(
        const yandex::solomon::config::rpc::TGrpcClientConfig& conf,
        NMonitoring::TMetricRegistry& registry,
        TString clientId)
{
    auto threadPool = CreateGrpcThreadPool(conf);
    auto sc = CreateGrpcServiceConnection<SlicerService>(
        conf,
        false,
        registry,
        std::move(threadPool),
        std::move(clientId));

    return CreateSlicerClient(registry, std::move(sc));
}

ISlicerClusterClientPtr CreateSlicerGrpcClusterClient(
        const yandex::solomon::config::rpc::TGrpcClientConfig& conf,
        NMonitoring::TMetricRegistry& registry,
        TString clientId)
{
    return std::make_shared<TGrpcClusterClientBase<SlicerService, ISlicerClient>>(
        conf,
        false,
        registry,
        CreateSlicerClient,
        std::move(clientId));
}

} // namespace NSolomon::NSlicerClient
