#include <library/cpp/testing/gtest/gtest.h>
#include <solomon/libs/cpp/clients/slicer/slicelet_listener.cpp>
#include <utility>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>
#include <solomon/libs/cpp/clients/slicer/slicelet_listener.h>
#include <solomon/libs/cpp/clients/slicer/slicer_client.h>

using namespace NActors;
using namespace NGrpc;
using namespace NSolomon::NSlicerClient;
using namespace NThreading;

namespace {

class TFailingClient: public ISlicerClient {
public:
    TAsyncGetAllAssignmentsResponse GetAllAssignments(TString) noexcept override {
        Y_FAIL("should not be used");
    }

    TAsyncGetSlicesByHostResponse GetSlicesByHost(TString, TString) noexcept override {
        ++CallsCnt;

        auto p = NewPromise<TGetSlicesByHostResponseOrErrorPtr>();
        auto status = TGrpcStatus::Internal("err");

        p.SetValue(std::make_unique<TGetSlicesByHostResponseOrError>(std::move(status)));
        return p.GetFuture();
    }

    void Stop(bool) override {
    }

public:
    size_t CallsCnt{0};
};

enum class EIsLeader {
    No = 0,
    Yes,
};

/**
 * A client which actually returns a response
 */
class TGoodClient: public ISlicerClient {
public:
    explicit TGoodClient(EIsLeader isLeader = EIsLeader::Yes, TString leaderAddress = {})
        : IsLeader_{isLeader}
        , LeaderAddress_{std::move(leaderAddress)}
    {}

private:
    TAsyncGetAllAssignmentsResponse GetAllAssignments(TString) noexcept override {
        Y_FAIL("should not be used");
    }

    TAsyncGetSlicesByHostResponse GetSlicesByHost(TString, TString) noexcept override {
        ++CallsCnt;

        auto p = NewPromise<TGetSlicesByHostResponseOrErrorPtr>();
        yandex::monitoring::slicer::GetSlicesByHostResponse resp;

        resp.set_is_leader(IsLeader_ == EIsLeader::Yes);
        resp.set_leader_address(LeaderAddress_);

        p.SetValue(std::make_unique<TGetSlicesByHostResponseOrError>(std::move(resp)));
        return p.GetFuture();
    }

    void Stop(bool) override {
    }

public:
    size_t CallsCnt{0};

private:
    EIsLeader IsLeader_;
    TString LeaderAddress_;
};

struct TAddressToClient {
    TString Address;
    std::shared_ptr<ISlicerClient> ClientPtr;
};

class TMockSlicerClusterClient: public NSolomon::NSlicerClient::ISlicerClusterClient {
public:
    explicit TMockSlicerClusterClient(TVector<TAddressToClient>& clients) {
        Addresses_.reserve(clients.size());

        for (auto& addressToClient: clients) {
            Clients_[addressToClient.Address] = addressToClient.ClientPtr;
            Addresses_.emplace_back(addressToClient.Address);
        }

        CurrentClientAddress = clients[0].Address;
        CurrentSlicerClient = clients[0].ClientPtr;
        CurrentIdx = Addresses_.size() - 1; // to start from zero on the first iteration, cause it'll be ++idx % size()
    }

    ISlicerClient* Get(TStringBuf addr) noexcept override {
        ++SwitchToCnt;
        CurrentClientAddress = addr;
        CurrentSlicerClient = Clients_[addr];

        return CurrentSlicerClient.get();
    }

    ISlicerClient* GetAny() noexcept override {
        return nullptr;
    }

    TString GetAnyAddress() const noexcept override {
        ++GetRandomAddressCnt;

        if (++CurrentIdx == Addresses_.size()) {
            CurrentIdx = 0;
        }

        CurrentClientAddress = Addresses_[CurrentIdx];
        CurrentSlicerClient = Clients_.at(CurrentClientAddress);

        return CurrentClientAddress;
    }

    const std::vector<TString>& Addresses() const noexcept override {
        return Addresses_;
    }

    void Add(TStringBuf) override {
        ythrow yexception() << "not implemented";
    }

    void Stop(bool) override {
    }

public:
    mutable size_t CurrentIdx{0};
    mutable size_t GetRandomAddressCnt{0};
    mutable size_t SwitchToCnt{0};
    mutable std::shared_ptr<ISlicerClient> CurrentSlicerClient;
    mutable TString CurrentClientAddress;

private:
    TVector<TString> Addresses_;
    THashMap<TString, std::shared_ptr<ISlicerClient>> Clients_;
};

} // namespace

class TSliceletListenerTest: public ::testing::Test {
protected:
    void SetUp() override {
        Runtime_ = NSolomon::TTestActorRuntime::CreateInited(1, false, true);
        Runtime_->WaitForBootstrap();

        EdgeId_ = Runtime_->AllocateEdgeActor();
        Runtime_->SetLogPriority(NSolomon::Wal, NActors::NLog::PRI_DEBUG);
    }

    void TearDown() override {
        Runtime_.Reset();
    }

    THolder<NSolomon::TTestActorRuntime> Runtime_;
    TActorId EdgeId_;
};

TEST_F(TSliceletListenerTest, HappyPath) {
    auto good1 = std::make_shared<TGoodClient>(EIsLeader::Yes);
    auto good2 = std::make_shared<TGoodClient>(EIsLeader::No);

    TVector<TAddressToClient> clients = {
            { "good_1", good1 },
            { "good_2", good2 },
    };

    auto clusterClient = std::make_shared<TMockSlicerClusterClient>(clients);

    TSliceletListenerOptions opts{
            .Service = "someService",
            .Host = "localhost",
            .UpdateInterval = TDuration::MilliSeconds(10),
            .RetriesBackoff = TDuration::Zero(),
            .MaxRetries = 0,
            .SlicerClients = clusterClient,
    };

    auto listener = Runtime_->Register(CreateSliceletListenerActor(opts));
    Runtime_->WaitForBootstrap();
    Runtime_->Send(new IEventHandle(listener, EdgeId_, new TSliceletListenerEvents::TEvSubscribe));

    auto eventPtr = Runtime_->GrabEdgeEvent<TSliceletListenerEvents::TSlicesUpdate>(EdgeId_, TDuration::Max());

    EXPECT_EQ(clusterClient->GetRandomAddressCnt, 1ul);
    EXPECT_EQ(clusterClient->SwitchToCnt, 1ul);
    EXPECT_EQ(clusterClient->CurrentClientAddress, clients[0].Address);
    EXPECT_EQ(clusterClient->CurrentSlicerClient, clients[0].ClientPtr);

    EXPECT_EQ(good1->CallsCnt, 1ul);
    EXPECT_EQ(good2->CallsCnt, 0ul);
}

TEST_F(TSliceletListenerTest, ClientsSwitching) {
    auto failingClient = std::make_shared<TFailingClient>();
    auto goodClient = std::make_shared<TGoodClient>();

    TVector<TAddressToClient> clients = {
            { "FailingClient", failingClient},
            { "GoodClient", goodClient},
    };
    auto clusterClient = std::make_shared<TMockSlicerClusterClient>(clients);

    TSliceletListenerOptions opts{
            .Service = "someService",
            .Host = "localhost",
            .UpdateInterval = TDuration::MilliSeconds(10),
            .RetriesBackoff = TDuration::Zero(),
            .MaxRetries = 0,
            .SlicerClients = clusterClient,
    };

    auto listener = Runtime_->Register(CreateSliceletListenerActor(opts));
    Runtime_->WaitForBootstrap();
    Runtime_->Send(new IEventHandle(listener, EdgeId_, new TSliceletListenerEvents::TEvSubscribe));

    auto eventPtr = Runtime_->GrabEdgeEvent<TSliceletListenerEvents::TSlicesUpdate>(EdgeId_, TDuration::Max());

    EXPECT_EQ(clusterClient->GetRandomAddressCnt, 2ul);
    EXPECT_EQ(clusterClient->SwitchToCnt, 2ul);
    EXPECT_EQ(clusterClient->CurrentClientAddress, clients[1].Address);
    EXPECT_EQ(clusterClient->CurrentSlicerClient, clients[1].ClientPtr);

    EXPECT_EQ(failingClient->CallsCnt, 1ul);
    EXPECT_EQ(goodClient->CallsCnt, 1ul);
}

TEST_F(TSliceletListenerTest, SwitchingToALeader) {
    auto good1 = std::make_shared<TGoodClient>(EIsLeader::No, /*leaderAddress = */"good_2");
    auto good2 = std::make_shared<TGoodClient>(EIsLeader::Yes);

    TVector<TAddressToClient> clients = {
            { "good_1", good1 },
            { "good_2", good2},
    };

    auto clusterClient = std::make_shared<TMockSlicerClusterClient>(clients);

    TSliceletListenerOptions opts{
            .Service = "someService",
            .Host = "localhost",
            .UpdateInterval = TDuration::MilliSeconds(10),
            .RetriesBackoff = TDuration::Zero(),
            .MaxRetries = 0,
            .SlicerClients = clusterClient,
    };

    auto listener = Runtime_->Register(CreateSliceletListenerActor(opts));
    Runtime_->WaitForBootstrap();
    Runtime_->Send(new IEventHandle(listener, EdgeId_, new TSliceletListenerEvents::TEvSubscribe));

    auto eventPtr = Runtime_->GrabEdgeEvent<TSliceletListenerEvents::TSlicesUpdate>(EdgeId_, TDuration::Max());

    EXPECT_EQ(clusterClient->GetRandomAddressCnt, 1ul);
    EXPECT_EQ(clusterClient->SwitchToCnt, 2ul);
    EXPECT_EQ(clusterClient->CurrentClientAddress, clients[1].Address);
    EXPECT_EQ(clusterClient->CurrentSlicerClient, clients[1].ClientPtr);

    EXPECT_EQ(good1->CallsCnt, 1ul);
    EXPECT_EQ(good2->CallsCnt, 1ul);
}

TEST_F(TSliceletListenerTest, Retries) {
    auto failingClient = std::make_shared<TFailingClient>();
    auto goodClient = std::make_shared<TGoodClient>();

    TVector<TAddressToClient> clients = {
            { "FailingClient", failingClient },
            { "GoodClient", goodClient },
    };
    auto clusterClient = std::make_shared<TMockSlicerClusterClient>(clients);

    TSliceletListenerOptions opts{
            .Service = "someService",
            .Host = "localhost",
            .UpdateInterval = TDuration::MilliSeconds(10),
            .RetriesBackoff = TDuration::Zero(),
            .MaxRetries = 2,
            .SlicerClients = clusterClient,
    };

    auto listener = Runtime_->Register(CreateSliceletListenerActor(opts));
    Runtime_->WaitForBootstrap();
    Runtime_->Send(new IEventHandle(listener, EdgeId_, new TSliceletListenerEvents::TEvSubscribe));

    auto eventPtr = Runtime_->GrabEdgeEvent<TSliceletListenerEvents::TSlicesUpdate>(EdgeId_, TDuration::Max());

    EXPECT_EQ(clusterClient->GetRandomAddressCnt, 2ul);
    EXPECT_EQ(clusterClient->SwitchToCnt, 2ul);
    EXPECT_EQ(clusterClient->CurrentClientAddress, clients[1].Address);
    EXPECT_EQ(clusterClient->CurrentSlicerClient, clients[1].ClientPtr);

    EXPECT_EQ(failingClient->CallsCnt, 1ul + opts.MaxRetries);
    EXPECT_EQ(goodClient->CallsCnt, 1ul);
}
