#include <solomon/libs/cpp/clients/tsdb/rpc.h>

#include <infra/yasm/zoom/components/serialization/zoom_to_protobuf/value_series.h>

#include <library/cpp/colorizer/colors.h>
#include <library/cpp/getopt/small/last_getopt.h>
#include <library/cpp/getopt/small/modchooser.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/stream/file.h>
#include <util/generic/guid.h>

using namespace NSolomon;
using namespace NYasm::NInterfaces::NInternal;

class TCommandBase: public TMainClassArgs {
protected:
    void RegisterOptions(NLastGetopt::TOpts& opts) override {
        TMainClassArgs::RegisterOptions(opts);

        opts.SetFreeArgsNum(0);
        opts.AddLongOption("host")
            .RequiredArgument("<host>")
            .StoreResult(&Host_)
            .Required()
            .Help("hostname a tsdb instance");
        opts.AddLongOption('v', "verbose")
            .StoreTrue(&Verbose_)
            .Help("print logs");
    }

    static void PrintError(const TRequestError& status) {
        Cerr << NColorizer::RED << "Error code " << status.Type() << ": " << NColorizer::RESET;
        Cerr << status.Message() << Endl;
    }

    static std::shared_ptr<NTsdb::ITsdbRpc> Client(TString host, IHttpClientPtr client, NMonitoring::TMetricRegistry& registry) {
        return NTsdb::CreateNodeHttp(std::move(host), std::move(client), "tsdb-cli", {}, registry);
    }

protected:
    TString Host_;
    bool Verbose_ = false;
};

class TReadBase: public TCommandBase {
protected:
    void RegisterOptions(NLastGetopt::TOpts& opts) override {
        TCommandBase::RegisterOptions(opts);

        opts.AddLongOption("hosts")
            .RequiredArgument("<hosts>")
            .Help("host or metagroup specifier")
            .Required()
            .StoreResult(&Hosts_);

        opts.AddLongOption("tags")
            .RequiredArgument("<tags>")
            .Help("semicolon-separated list of additional tags, e.g. itype=upper;geo=sas")
            .StoreResult(&Tags_);

        opts.AddLongOption("signal")
            .RequiredArgument("<signal>")
            .Help("name of the signal")
            .Required()
            .StoreResult(&Signal_);

        opts.AddLongOption("from")
            .RequiredArgument("<from_seconds>")
            .Help("the beginning of the time range")
            .StoreResult(&From_);

        opts.AddLongOption("to")
            .RequiredArgument("<to_seconds>")
            .Help("the ending of the time range")
            .StoreResult(&To_);

        opts.AddLongOption("last")
            .RequiredArgument("<last_seconds>")
            .Help("time range [now-last, now]")
            .StoreResult(&Last_);

        opts.AddLongOption("period")
            .RequiredArgument("<period_seconds>")
            .Help("interval between points")
            .Required()
            .StoreResult(&Period_);

        opts.AddLongOption("replica")
            .RequiredArgument("<replica>")
            .Help("replica selector: 0 - ANY, 1/2 - FIRST/SECOND")
            .StoreResult(&Replica_);
    }

protected:
    TString Hosts_;
    TString Tags_;
    TString Signal_;
    ui64 From_ = 0;
    ui64 To_ = 0;
    ui64 Last_ = 0;
    ui64 Period_ = 0;
    int Replica_ = 0;

    static constexpr ui64 DEFAULT_PERIOD = 5;
};

class TRead: public TReadBase {
protected:
    void RegisterOptions(NLastGetopt::TOpts& opts) override {
        TReadBase::RegisterOptions(opts);

        opts.SetTitle("read all metrics that correspond to the given selectors");
    }

    int DoRun(NLastGetopt::TOptsParseResult&&) override {
        THistoryReadAggregatedRequest req;

        TGUID guid;
        CreateGuid(&guid);
        TString reqid{GetGuidAsString(guid)};

        if (Verbose_) {
            Cout << "Request id: " << reqid << Endl;
        }
        req.SetRequestId(reqid);
        TInstant deadline = TInstant::Now() + TDuration::Seconds(30);
        req.SetDeadlineTimestamp(deadline.Seconds());

        auto* query = req.AddQueries();
        req.MutableHostNameTable()->AddName(Hosts_);
        query->MutableHostName()->SetIndex(0);
        query->SetDontAggregate(true);

        if (Tags_) {
            req.MutableRequestKeyTable()->AddName(Tags_);
            query->MutableRequestKey()->SetIndex(0);
        }

        req.MutableSignalNameTable()->AddName(Signal_);
        query->MutableSignalName()->SetIndex(0);

        if (To_ == 0) {
            To_ = TInstant::Now().Seconds();
        }

        if (Period_ % DEFAULT_PERIOD != 0) {
            Cerr << NColorizer::RED << "Period must be divisible by " << DEFAULT_PERIOD << " seconds" << NColorizer::RESET << Endl;
            return 1;
        }

        if (Period_ == 0) {
            Period_ = DEFAULT_PERIOD;
        }

        To_ -= To_ % Period_;

        if ((From_ == 0) == (Last_ == 0)) {
            Cerr << NColorizer::RED << "Need to specify either --from or --last" << NColorizer::RESET << Endl;
            return 1;
        }

        if (From_ == 0) {
            From_ = To_ - Last_;
        }

        From_ -= From_ % Period_;

        query->SetStartTimestamp(From_);
        query->SetEndTimestamp(To_);

        query->SetPeriod(Period_);

        switch (Replica_) {
            case 0:
                break;
            case 1:
                query->SetReplicaSelector(THistoryAggregatedQuery_EReplicaSelector_FIRST);
                break;
            case 2:
                query->SetReplicaSelector(THistoryAggregatedQuery_EReplicaSelector_SECOND);
                break;
            default: {
                Cerr << NColorizer::RED << "Unsupported replica value " << Replica_ << Endl;
                return 1;
            }
        }

        TCurlClientOptions curlOpts;
        curlOpts.Name = "tsdb-client";
        curlOpts.WorkerThreads = 2;
        curlOpts.HandlerThreads = 2;

        // TODO(ivanzhukov): change to a better value
        curlOpts.DnsCacheLifetime = TDuration::Max();

        curlOpts.MaxInflight = 1000;
        curlOpts.QueueSizeLimit = 1000;

        auto& registry = *NMonitoring::TMetricRegistry::Instance();

        auto curlClient = CreateCurlClient(curlOpts, registry);

        auto res = Client(Host_, curlClient, registry)->ReadAggregated(req).ExtractValueSync();

        if (res.Fail()) {
            PrintError(res.Error());
            return 1;
        }

        auto& val = res.Value();
        Cerr << NColorizer::GREEN << "Ok" << NColorizer::RESET << Endl;
        if (val.SeriesSize() == 0) {
            Cerr << "no metrics found" << Endl;
            return 0;
        }

        auto& HostsPool = val.GetHostNameTable();
        auto& TagsPool = val.GetRequestKeyTable();
        auto& SignalPool = val.GetSignalNameTable();
        Y_UNUSED(HostsPool, TagsPool, SignalPool);

        ui32 index = 0;
        for (auto& series: val.GetSeries()) {
            Cout << "Series " << index++ << ", status " << static_cast<int>(series.GetStatusCode()) << Endl;
            Cout << "  Stating timestamp: " << series.GetStartTimestamp() << Endl;
            Cout << "  hosts=" << HostsPool.GetName(series.GetQuery().GetHostName().GetIndex()) << Endl;
            Cout << "  " << TagsPool.GetName(series.GetQuery().GetRequestKey().GetIndex()) << Endl;
            Cout << "  signal=" << SignalPool.GetName(series.GetQuery().GetSignalName().GetIndex()) << Endl;
            NZoom::NProtobuf::TValueSeriesDeserializer::DeserializeAndVisit(series.GetValueSeries(), [] (auto&& value) {
                Cout << "  " << value << Endl;
            });
        }

        return 0;
    }
};

class TFetchHosts : public TCommandBase {
protected:
    void RegisterOptions(NLastGetopt::TOpts& opts) override {
        TCommandBase::RegisterOptions(opts);

        opts.AddLongOption("group")
                .RequiredArgument("<group>")
                .Help("group specifier (e.g. SAS.000)")
                .Required()
                .StoreResult(&Group_);

        opts.SetTitle("fetch hosts processed by this node in a specified group");
    }

    int DoRun(NLastGetopt::TOptsParseResult&&) override {
        auto& registry = *NMonitoring::TMetricRegistry::Instance();

        TCurlClientOptions curlOpts;
        curlOpts.Name = "tsdb-client";
        curlOpts.WorkerThreads = 2;
        curlOpts.HandlerThreads = 2;

        // TODO(ivanzhukov): change to a better value
        curlOpts.DnsCacheLifetime = TDuration::Max();

        curlOpts.MaxInflight = 1000;
        curlOpts.QueueSizeLimit = 1000;

        auto curlClient = CreateCurlClient(curlOpts, registry);

        auto res = Client(Host_, curlClient, registry)->FetchHosts(Group_).ExtractValueSync();

        if (res.Fail()) {
            PrintError(res.Error());
            return 1;
        }

        const auto& hosts = res.Value().Hosts;

        Cerr << NColorizer::GREEN << "Ok" << NColorizer::RESET << Endl;
        if (hosts.empty()) {
            Cerr << "no hosts found" << Endl;
            return 0;
        }

        size_t i = 0;
        Cout << '[';
        for (const auto& host : hosts) {
            if (i++) {
                Cout << ", ";
            }
            Cout << '"' << host << '"';
        }
        Cout << ']' << Endl;

        return 0;
    }

private:
    TString Group_;
};

class TMain: public TMainClassModes {
protected:
    void RegisterModes(TModChooser& modes) override {
        TMainClassModes::RegisterModes(modes);

        modes.SetDescription("tsdb client");

        modes.AddGroupModeDescription("read");

        modes.AddMode("read", new TRead{}, "read metrics");
        modes.AddMode("fetch_hosts", new TFetchHosts{}, "fetch hosts processed by this node in a specified group");

        modes.AddCompletions("tsdb-cli");
    }
};

int main(int argc, const char** argv) {
    return TMain{}.Run(argc, argv);
}
