#include "counters.h"

#include <util/string/cast.h>

namespace NSolomon {
void TBaseHttpCallCounters::ReportCallStart() {
    Started_->Inc();
    Inflight_->Inc();
}

void TBaseHttpCallCounters::ReportCallStats(HttpCodes code, TDuration duration) {
    Timings_->Record(duration.MilliSeconds());
    code == HTTP_OK ? CompletedOk_->Inc() : CompletedError_->Inc();
    Inflight_->Dec();
    ResolveStatus(code)->Inc();
}

void TBaseHttpCallCounters::ReportInterruption(TString reason, TDuration duration) {
    Started_->Inc();
    Timings_->Record(duration.MilliSeconds());
    CompletedError_->Inc();
    ResolveReason(reason)->Inc();
}

NMonitoring::ILabelsPtr TBaseHttpCallCounters::MakeLabels(TStringBuf name, const NMonitoring::TLabels& additional) {
    NMonitoring::ILabelsPtr result{new NMonitoring::TLabels};
    TString fullName = Prefix_ + name;
    result->Add("sensor", fullName);
    for (const auto& l: CommonLabels_) {
        result->Add(l);
    }
    for (const auto& l: additional) {
        result->Add(l);
    }
    return result;
}

NMonitoring::IRate* TBaseHttpCallCounters::ResolveStatus(HttpCodes code) {
    {
        auto status = Status_.Read();
        if (auto it = status->find(code); it != status->end()) {
            return it->second;
        }
    }

    auto* metric = Registry_.Rate(MakeLabels("call.status", {{"status", ToString(static_cast<int>(code))}}));
    {
        auto status = Status_.Write();
        (*status)[code] = metric;
    }
    return metric;
}

NMonitoring::IRate* TBaseHttpCallCounters::ResolveReason(TString reason) {
    {
        auto errors = SpecialErrors_.Read();
        if (auto it = errors->find(reason); it != errors->end()) {
            return it->second;
        }
    }

    auto* metric = Registry_.Rate(MakeLabels("call.status", {{"status", reason}}));
    {
        auto error = SpecialErrors_.Write();
        (*error)[reason] = metric;
    }
    return metric;
}
} // namespace NSolomon
