#pragma once

#include <solomon/libs/cpp/sync/rw_lock.h>

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/http/misc/httpcodes.h>

#include <util/system/hp_timer.h>


namespace NSolomon {
class IHttpCallCounters: public TThrRefBase {
public:
    virtual ~IHttpCallCounters() = default;
    virtual void ReportCallStart() = 0;
    virtual void ReportCallStats(HttpCodes code, TDuration duration) = 0;
    /**
     * For special error cases like CircuitBreaker interruption, that does not fit into http codes list.
     */
    virtual void ReportInterruption(TString reason, TDuration duration) = 0;
};

class TBaseHttpCallCounters: public IHttpCallCounters {
public:
    TBaseHttpCallCounters(NMonitoring::IMetricRegistry& registry, NMonitoring::TLabels labels, TString prefix)
        : Registry_{registry}
        , CommonLabels_{std::move(labels)}
        , Prefix_{std::move(prefix)}
    {
        Timings_ = registry.HistogramRate(
                MakeLabels("call.elapsedTimeMs"),
                NMonitoring::ExponentialHistogram(13, 2, 16));

        Started_ = registry.Rate(MakeLabels("call.started"));
        CompletedOk_ = registry.Rate(MakeLabels("call.completedOk"));
        CompletedError_ = registry.Rate(MakeLabels("call.completedError"));
        Inflight_ = registry.IntGauge(MakeLabels("call.inFlight"));
    }
    virtual ~TBaseHttpCallCounters() = default;

    void ReportCallStart() override;
    void ReportCallStats(HttpCodes code, TDuration duration) override;
    void ReportInterruption(TString reason, TDuration duration) override;

private:
    NMonitoring::ILabelsPtr MakeLabels(TStringBuf name, const NMonitoring::TLabels& additional = {});
    NMonitoring::IRate* ResolveStatus(HttpCodes code);
    NMonitoring::IRate* ResolveReason(TString reason);

private:
    NMonitoring::IMetricRegistry& Registry_;
    const NMonitoring::TLabels CommonLabels_;
    const TString Prefix_;
    NMonitoring::IHistogram* Timings_{};
    NMonitoring::IRate* Started_{};
    NMonitoring::IRate* CompletedOk_{};
    NMonitoring::IRate* CompletedError_{};
    NMonitoring::IIntGauge* Inflight_{};
    NSync::TLightRwLock<THashMap<HttpCodes, NMonitoring::IRate*>> Status_;
    NSync::TLightRwLock<THashMap<TString, NMonitoring::IRate*>> SpecialErrors_;
};

class TServerHttpCallCounters: public TBaseHttpCallCounters {
public:
    TServerHttpCallCounters(NMonitoring::IMetricRegistry& registry, const NMonitoring::TLabels& labels)
        : TBaseHttpCallCounters(registry, labels, "http.server.")
    {
    };
};

class TClientHttpCallCounters: public TBaseHttpCallCounters {
public:
    TClientHttpCallCounters(NMonitoring::IMetricRegistry& registry, const NMonitoring::TLabels& labels)
        : TBaseHttpCallCounters(registry, labels, "http.client.")
    {
    };
};

class TAggregateHttpCallCounters: public IHttpCallCounters {
public:
    TAggregateHttpCallCounters(TBaseHttpCallCounters* target, TBaseHttpCallCounters* total)
        : Target_(target)
        , Total_(total)
    {
    }

    void ReportCallStart() override {
        Target_->ReportCallStart();
        Total_->ReportCallStart();
    }

    void ReportCallStats(HttpCodes code, TDuration duration) override {
        Target_->ReportCallStats(code, duration);
        Total_->ReportCallStats(code, duration);
    }

private:
    TBaseHttpCallCounters* Target_;
    TBaseHttpCallCounters* Total_;
};
} // namespace NSolomon
