#include <solomon/libs/cpp/clients/tsdb/metrics/counters.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;

TEST(TGrpcCountersTest, ClientEmptyLabels) {
    NMonitoring::TMetricRegistry registry;
    auto metrics = TClientHttpCallCounters(registry, NMonitoring::TLabels());

    auto started = registry.Rate({{"sensor", "http.client.call.started"}});
    auto completedOk = registry.Rate({{"sensor", "http.client.call.completedOk"}});
    auto completedError = registry.Rate({{"sensor", "http.client.call.completedError"}});
    auto inFlight = registry.IntGauge({{"sensor", "http.client.call.inFlight"}});

    EXPECT_EQ(0u, started->Get());
    EXPECT_EQ(0u, completedOk->Get());
    EXPECT_EQ(0u, completedError->Get());
    EXPECT_EQ(0, inFlight->Get());

    metrics.ReportCallStart();
    EXPECT_EQ(1u, started->Get());
    EXPECT_EQ(0u, completedOk->Get());
    EXPECT_EQ(0u, completedError->Get());
    EXPECT_EQ(1, inFlight->Get());

    metrics.ReportCallStart();
    EXPECT_EQ(2u, started->Get());
    EXPECT_EQ(0u, completedOk->Get());
    EXPECT_EQ(0u, completedError->Get());
    EXPECT_EQ(2, inFlight->Get());

    metrics.ReportCallStats(HTTP_OK, TDuration::MilliSeconds(10));
    EXPECT_EQ(2u, started->Get());
    EXPECT_EQ(1u, completedOk->Get());
    EXPECT_EQ(0u, completedError->Get());
    EXPECT_EQ(1, inFlight->Get());

    metrics.ReportCallStats(HTTP_INTERNAL_SERVER_ERROR, TDuration::MilliSeconds(5));
    EXPECT_EQ(2u, started->Get());
    EXPECT_EQ(1u, completedOk->Get());
    EXPECT_EQ(1u, completedError->Get());
    EXPECT_EQ(0, inFlight->Get());
}

TEST(TGrpcCountersTest, ClientLables) {
    NMonitoring::TMetricRegistry registry;
    auto metrics = TClientHttpCallCounters(registry, {{"myKey", "myValue"}});

    auto started = registry.Rate({{"sensor", "http.client.call.started"}, {"myKey", "myValue"}});
    auto inFlight = registry.IntGauge({{"sensor", "http.client.call.inFlight"}, {"myKey", "myValue"}});
    for (int index = 1; index <= 10; index++) {
        metrics.ReportCallStart();
        EXPECT_EQ(static_cast<ui64>(index), started->Get());
        EXPECT_EQ(index, inFlight->Get());
    }
}

TEST(TGrpcCountersTest, ServerLables) {
    NMonitoring::TMetricRegistry registry;
    auto metrics = TServerHttpCallCounters(registry, {{"myKey", "myValue"}});

    auto started = registry.Rate({{"sensor", "http.server.call.started"}, {"myKey", "myValue"}});
    auto inFlight = registry.IntGauge({{"sensor", "http.server.call.inFlight"}, {"myKey", "myValue"}});
    for (int index = 1; index <= 10; index++) {
        metrics.ReportCallStart();
        EXPECT_EQ(static_cast<ui64>(index), started->Get());
        EXPECT_EQ(index, inFlight->Get());
    }

    auto completedError = registry.Rate({{"sensor", "http.server.call.completedError"}, {"myKey", "myValue"}});
    metrics.ReportCallStats(HTTP_BAD_REQUEST, TDuration::MilliSeconds(5));
    EXPECT_EQ(1u, completedError->Get());
    EXPECT_EQ(9, inFlight->Get());
}

TEST(TGrpcCountersTest, Aggregate) {
    NMonitoring::TMetricRegistry registry;
    auto alice = TServerHttpCallCounters(registry, {{"target", "alice"}});
    auto bob = TServerHttpCallCounters(registry, {{"target", "bob"}});
    auto aggr = TServerHttpCallCounters(registry, {{"target", "total"}});

    {   // 2 started, 1 completed
        auto metrics = TAggregateHttpCallCounters(&alice, &aggr);
        metrics.ReportCallStart();
        metrics.ReportCallStart();
        metrics.ReportCallStats(HTTP_OK, TDuration::MilliSeconds(5));
    }

    {   // 3 started, 1 completed
        auto metrics = TAggregateHttpCallCounters(&bob, &aggr);
        metrics.ReportCallStart();
        metrics.ReportCallStats(HTTP_OK, TDuration::MilliSeconds(5));
        metrics.ReportCallStart();
        metrics.ReportCallStart();
    }

    EXPECT_EQ(2u, registry.Rate({{"sensor", "http.server.call.started"}, {"target", "alice"}})->Get());
    EXPECT_EQ(3u, registry.Rate({{"sensor", "http.server.call.started"}, {"target", "bob"}})->Get());
    EXPECT_EQ(5u, registry.Rate({{"sensor", "http.server.call.started"}, {"target", "total"}})->Get());

    EXPECT_EQ(1u, registry.Rate({{"sensor", "http.server.call.completedOk"}, {"target", "alice"}})->Get());
    EXPECT_EQ(1u, registry.Rate({{"sensor", "http.server.call.completedOk"}, {"target", "bob"}})->Get());
    EXPECT_EQ(2u, registry.Rate({{"sensor", "http.server.call.completedOk"}, {"target", "total"}})->Get());

    EXPECT_EQ(1, registry.IntGauge({{"sensor", "http.server.call.inFlight"}, {"target", "alice"}})->Get());
    EXPECT_EQ(2, registry.IntGauge({{"sensor", "http.server.call.inFlight"}, {"target", "bob"}})->Get());
    EXPECT_EQ(3, registry.IntGauge({{"sensor", "http.server.call.inFlight"}, {"target", "total"}})->Get());
}

TEST(TGrpcCountersTest, StatusCount) {
    NMonitoring::TMetricRegistry registry;
    auto metrics = TClientHttpCallCounters(registry, NMonitoring::TLabels());

    metrics.ReportCallStart();
    metrics.ReportCallStats(HTTP_OK, TDuration::MilliSeconds(10));

    auto ok = registry.Rate({{"sensor", "http.client.call.status"}, {"status", "200"}});
    auto gone = registry.Rate({{"sensor", "http.client.call.status"}, {"status", "410"}});
    EXPECT_EQ(1u, ok->Get());
    EXPECT_EQ(0u, gone->Get());

    metrics.ReportCallStart();
    metrics.ReportCallStats(HTTP_OK, TDuration::MilliSeconds(10));
    EXPECT_EQ(2u, ok->Get());
    EXPECT_EQ(0u, gone->Get());

    metrics.ReportCallStart();
    metrics.ReportCallStats(HTTP_GONE, TDuration::MilliSeconds(10));
    EXPECT_EQ(2u, ok->Get());
    EXPECT_EQ(1u, gone->Get());
}
