#include "rpc.h"

#include <solomon/libs/cpp/circuit_breaker/circuit_breaker.h>
#include <solomon/libs/cpp/clients/tsdb/metrics/counters.h>
#include <solomon/libs/cpp/config/units.h>
#include <solomon/libs/cpp/http/client/curl/client.h>
#include <solomon/libs/cpp/string_map/string_map.h>
#include <solomon/libs/cpp/sync/rw_lock.h>

#include <infra/yasm/interfaces/internal/history_api.pb.h>
#include <infra/yasm/server/lib/msgpack_tools.h>

#include <library/cpp/json/json_writer.h>
#include <library/cpp/threading/future/future.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/stream/str.h>
#include <util/string/builder.h>

#include <functional>

namespace NSolomon::NTsdb {

using namespace NSolomon;
using namespace NYasm::NInterfaces::NInternal;

namespace {

template <typename TProto>
TProto Deserialize(const TString &buffer) {
    TProto result;
    Y_ENSURE(result.ParseFromString(buffer), "Failed to parse as protobuf");
    return std::move(result);
}

template <>
TFetchHostsResponse Deserialize<TFetchHostsResponse>(const TString &buffer) {
    TFetchHostsResponse res;
    NJson::TJsonValue value = NYasmServer::MsgpackToJson(buffer);
    const auto& hosts = value["hosts"].GetArray();

    for (auto it = hosts.cbegin(); it != hosts.cend(); ++it) {
        res.Hosts.emplace_back(it->GetString());
    }

    return res;
}

bool TryAcquirePermission(const std::shared_ptr<TAtomicCircuitBreaker>& circuitBreaker) {
    return circuitBreaker->TryAcquirePermission(TInstant::Now());
}

void MarkFailure(const std::shared_ptr<TAtomicCircuitBreaker>& circuitBreaker, const TInstant& now) {
    return circuitBreaker->MarkFailure(now);
}

void MarkSuccess(const std::shared_ptr<TAtomicCircuitBreaker>& circuitBreaker, const TInstant& now) {
    return circuitBreaker->MarkSuccess(now);
}

bool NoOpTrue(const std::shared_ptr<TAtomicCircuitBreaker>&) {
    return true;
}

void NoOp(const std::shared_ptr<TAtomicCircuitBreaker>&, const TInstant&) {
    return;
}

class TTsdbRpc final: public ITsdbRpc {
public:
    TTsdbRpc(
            TString address,
            IHttpClientPtr httpClient,
            TString clientId,
            TRequestOpts requestOpts,
            NMonitoring::TMetricRegistry& registry,
            std::optional<TCircuitBreakerConfig> circuitBreakerConfig)
        : Address_(std::move(address))
        , HttpClient_(std::move(httpClient))
        , ClientId_(std::move(clientId))
        , RequestOptions_(std::move(requestOpts))
        , Registry_(registry)
        , CircuitBreaker_(
                circuitBreakerConfig ? std::make_shared<TAtomicCircuitBreaker>(
                        circuitBreakerConfig->GetFailureQuantileThreshold(),
                        FromProtoTime(circuitBreakerConfig->GetResetTimeout(), TDuration::Seconds(30)),
                        TInstant::Now())
                                     : nullptr)
        , AcquirePermission_(circuitBreakerConfig ? TryAcquirePermission : NoOpTrue)
        , MarkSuccess_(circuitBreakerConfig ? MarkSuccess : NoOp)
        , MarkFailure_(circuitBreakerConfig ? MarkFailure : NoOp)
    {
    }

    TReadAggregatedAsyncResponse ReadAggregated(const THistoryReadAggregatedRequest& request) override {
        return Request<THistoryReadAggregatedRequest, THistoryReadAggregatedResponse>(
                "/read_aggregated_protobuf",
                request,
                RequestOptions_);
    }

    // TODO(ivanzhukov): write to/out of msgpack directly
    TFetchHostsResponseAsyncResponse FetchHosts(TStringBuf group) override {
        TStringStream out;
        NJson::TJsonWriter json(&out, false);

        json.OpenMap();
        json.WriteKey("groups");
        json.OpenArray();
        json.Write(group);
        json.CloseArray();

        auto now = TInstant::Now();
        json.WriteKey("start");
        json.Write(now.Seconds() - 1800);
        json.WriteKey("end");
        json.Write(now.Seconds());
        json.CloseMap();

        json.Flush();

        return Request<TString, TFetchHostsResponse>("/fetch_hosts", out.Str(), RequestOptions_);
    }

    void Stop(bool) override {
        // nop
    }

private:
    template <typename TReq, typename TRes, typename TFutureResult = TErrorOr<TRes, TRequestError>>
    NThreading::TFuture<TFutureResult> Request(TStringBuf path, const TReq& request, const TRequestOpts& options) {
        auto promise = NThreading::NewPromise<TFutureResult>();

        if (!AcquirePermission_(CircuitBreaker_)) {
            auto metrics = GetOrCreateCounters(path);
            metrics->ReportInterruption("OPEN_CIRCUIT_BREAKER", TDuration::Zero());
            promise.SetValue(std::move(TRequestError{TRequestError::EType::Unknown, "Interrupted by CircuitBreaker"}));
            return promise.GetFuture();
        }

        TString url = TStringBuilder{} << Address_ << path;
        TString data;

        if constexpr (std::is_base_of_v<::google::protobuf::Message, TReq>) {
            Y_ENSURE(request.SerializeToString(&data), "Failed to serialize to protobuf");
        } else if constexpr (std::is_same_v<TString, TReq>) {
            data = std::move(NYasmServer::JsonToMsgpack(request));
        } else {
            static_assert(TDependentFalse<TReq>, "unsupported request type");
        }

        auto headers = Headers();
        headers->Add("User-Agent", ClientId_);
        if constexpr (std::is_base_of_v<::google::protobuf::Message, TReq>) {
            headers->Add("Content-Type", "application/x-protobuf");
            headers->Add("Accept", "application/x-protobuf");
        }

        auto metrics = GetOrCreateCounters(path);
        auto start = TInstant::Now();
        auto circuitBreaker = CircuitBreaker_;
        metrics->ReportCallStart();
        auto markSuccess = &MarkSuccess_;
        auto markFailure = &MarkFailure_;

        HttpClient_->Request(
                CreateRequest(EHttpMethod::Post, url, data, std::move(headers)),
                [promise, metrics, start, circuitBreaker, markSuccess, markFailure] (IHttpClient::TResult result) mutable {
                    try {
                        auto now = TInstant::Now();
                        if (!result.Success()) {
                            (*markFailure)(circuitBreaker, now);
                            promise.SetValue(std::move(TFutureResult::FromError(result.Error())));
                            metrics->ReportCallStats(HTTP_INTERNAL_SERVER_ERROR, now - start);
                            return;
                        }
                        auto response = result.Extract();
                        if (response->Code() != HTTP_OK) {
                            (*markFailure)(circuitBreaker, now);
                            promise.SetValue(std::move(TRequestError{
                                    TRequestError::EType::Unknown,
                                    TStringBuilder{} << "Server replied with " << response->Code()
                                                     << " code, message: " << response->ExtractData()}));
                            metrics->ReportCallStats(response->Code(), now - start);
                            return;
                        }
                        (*markSuccess)(circuitBreaker, now);
                        promise.SetValue(std::move(
                                TFutureResult::FromValue(Deserialize<TRes>(std::move(response->ExtractData())))));
                        metrics->ReportCallStats(response->Code(), now - start);
                    } catch (...) {
                        auto now = TInstant::Now();
                        (*markFailure)(circuitBreaker, now);
                        metrics->ReportCallStats(HTTP_INTERNAL_SERVER_ERROR, now - start);
                        promise.SetValue(
                                std::move(TRequestError{TRequestError::EType::Unknown, CurrentExceptionMessage()}));
                    }
                },
                options);

        return promise.GetFuture();
    }

    TIntrusivePtr<TClientHttpCallCounters> GetOrCreateCounters(TStringBuf endpoint) {
        {
            auto counters = Counters_.Read();
            if (auto* callCounters = counters->FindPtr(endpoint)) {
                return callCounters->Get();
            }
        }
        {
            auto counters = Counters_.Write();
            auto& clientCallCounters = (*counters)[endpoint];
            if (!clientCallCounters) {
                clientCallCounters = MakeIntrusive<TClientHttpCallCounters>(
                        Registry_,
                        NMonitoring::TLabels{{"endpoint", endpoint}});
            }
            return clientCallCounters.Get();
        }
    }

private:
    TString Address_;
    IHttpClientPtr HttpClient_;
    TString ClientId_;
    TRequestOpts RequestOptions_;
    NMonitoring::TMetricRegistry& Registry_;
    NSync::TLightRwLock<THashMap<TString, TIntrusivePtr<TClientHttpCallCounters>>> Counters_;
    std::shared_ptr<TAtomicCircuitBreaker> CircuitBreaker_;
    std::function<bool(const std::shared_ptr<TAtomicCircuitBreaker>&)> AcquirePermission_;
    std::function<void(const std::shared_ptr<TAtomicCircuitBreaker>&, const TInstant&)> MarkSuccess_;
    std::function<void(const std::shared_ptr<TAtomicCircuitBreaker>&, const TInstant&)> MarkFailure_;
};

class TTsdbClusterRpc: public ITsdbClusterRpc {
public:
    TTsdbClusterRpc(
            ui32 port,
            TString clientId,
            TTsdbClientConfig config,
            NMonitoring::TMetricRegistry& registry,
            TCircuitBreakerConfig circuitBreakerConfig)
        : Port_{port}
        , ClientId_{std::move(clientId)}
        , RequestOptions_{std::move(config.RequestOptions)}
        , Registry_{registry}
        , HttpClient_{CreateCurlClient(std::move(config.CurlClientOptions), Registry_)}
        , CircuitBreakerConfig_{std::move(circuitBreakerConfig)}
    {
    }

private:
    void Add(TStringBuf host) override {
        auto& client = Clients_[host];
        if (client) {
            return;
        }

        TStringBuilder sb;
        sb << host << ':' << Port_;
        client = CreateNodeHttp(std::move(sb), HttpClient_, ClientId_, RequestOptions_, Registry_, CircuitBreakerConfig_);
    }

    void Stop(bool) override {
    }

    ITsdbRpc* Get(TStringBuf host) noexcept override {
        if (auto it = Clients_.find(host); it != Clients_.end()) {
            return it->second.get();
        } else {
            return nullptr;
        }
    }

private:
    ui32 Port_;
    TString ClientId_;
    TRequestOpts RequestOptions_;
    NMonitoring::TMetricRegistry& Registry_;
    IHttpClientPtr HttpClient_;
    TStringMap<std::unique_ptr<ITsdbRpc>> Clients_;
    TCircuitBreakerConfig CircuitBreakerConfig_;
};

} // namespace

std::unique_ptr<ITsdbRpc> CreateNodeHttp(
        TString address,
        IHttpClientPtr httpClient,
        TString clientId,
        TRequestOpts requestOpts,
        NMonitoring::TMetricRegistry& registry,
        const std::optional<TCircuitBreakerConfig>& circuitBreakerConfig)
{
    Y_VERIFY(httpClient);

    return std::make_unique<TTsdbRpc>(
            std::move(address),
            std::move(httpClient),
            std::move(clientId),
            std::move(requestOpts),
            registry,
            circuitBreakerConfig);
}

std::shared_ptr<ITsdbClusterRpc> CreateTsdbClusterRpc(
        ui32 port,
        TString clientId,
        TTsdbClientConfig config,
        NMonitoring::TMetricRegistry& registry,
        TCircuitBreakerConfig circuitBreakerConfig)
{

    return std::make_shared<TTsdbClusterRpc>(
            port,
            std::move(clientId),
            std::move(config),
            registry,
            std::move(circuitBreakerConfig));
}

} // namespace NSolomon::NTsdb
