#include "access_service.h"

#include <solomon/libs/cpp/grpc/client/client.h>

#include <library/cpp/grpc/client/grpc_client_low.h>

namespace NSolomon {
namespace {

using namespace yandex::cloud::priv::servicecontrol::v1;
using NMonitoring::TMetricRegistry;
using yandex::solomon::config::rpc::TGrpcClientConfig;

TErrorOr<TIamAccount, TString> SubjectFromProto(const Subject& subj) {
    if (subj.has_service_account()) {
        return TIamAccount{
            .Type = EIamAccountType::Service,
            .Id = subj.service_account().id(),
            .ScopeId = subj.service_account().folder_id(),
        };
    } else if (subj.has_user_account()) {
        return TIamAccount{
            .Type = EIamAccountType::User,
            .Id = subj.user_account().id(),
            .ScopeId = subj.user_account().federation_id(),
        };
    } else if (subj.has_anonymous_account()) {
        return TIamAccount{
            .Type = EIamAccountType::Anonymous,
        };
    } else {
        return TString{"unknown Subject format"};
    }
}

class TGrpcAccessServiceClient: public IAccessServiceClient {
public:
    explicit TGrpcAccessServiceClient(std::unique_ptr<TGrpcServiceConnection<AccessService>> connection)
        : Connection_{std::move(connection)}
    {
    }

    TAsyncAuthenticateResponse Authenticate(TString iamToken) noexcept override {
        AuthenticateRequest req;
        req.set_iam_token(iamToken);

        auto promise = NThreading::NewPromise<TAsyncAuthenticateResponse::value_type>();

        auto cb = [promise] (NGrpc::TGrpcStatus&& status, AuthenticateResponse&& result) mutable {
            if (!status.Ok()) {
                TString errMsg = TStringBuilder() << "(" << status.GRpcStatusCode << "): " << status.Msg;
                // TODO(ivanzhukov@): map more codes
                EAccessServiceAuthErrorType errType = (status.GRpcStatusCode == grpc::StatusCode::DEADLINE_EXCEEDED)
                        ? EAccessServiceAuthErrorType::InternalNonRetriable
                        : EAccessServiceAuthErrorType::InternalRetriable;

                promise.SetValue(TAuthError{errType, std::move(errMsg)});
                return;
            }

            if (!result.has_subject()) {
                promise.SetValue(TAuthError{
                        EAccessServiceAuthErrorType::InternalNonRetriable,
                        "unknown response format"});
                return;
            }

            auto accountOrError = SubjectFromProto(result.subject());
            if (accountOrError.Fail()) {
                promise.SetValue(TAuthError{EAccessServiceAuthErrorType::InternalNonRetriable, accountOrError.Error()});
            } else {
                promise.SetValue(std::move(accountOrError.Value()));
            }
        };

        Connection_->Request<AuthenticateRequest, AuthenticateResponse>(
                req,
                std::move(cb),
                &AccessService::Stub::AsyncAuthenticate);

        return promise.GetFuture();
    }

    TAsyncAuthorizeResponse Authorize(const AuthorizeRequest& req) noexcept override {
        auto promise = NThreading::NewPromise<TAsyncAuthorizeResponse::value_type>();

        auto cb = [promise] (NGrpc::TGrpcStatus&& status, AuthorizeResponse&& result) mutable {
            if (!status.Ok()) {
                promise.SetValue(status);
                return;
            }

            promise.SetValue(std::move(result));
        };

        Connection_->Request<AuthorizeRequest, AuthorizeResponse>(
                req,
                std::move(cb),
                &AccessService::Stub::AsyncAuthorize);

        return promise.GetFuture();
    }

    void Stop(bool) override {
    }

private:
    std::unique_ptr<TGrpcServiceConnection<AccessService>> Connection_;
};

} // namespace

IAccessServiceClientPtr CreateAccessServiceGrpcClient(
        const yandex::solomon::config::rpc::TGrpcClientConfig& conf,
        NMonitoring::TMetricRegistry& registry,
        TString clientId)
{
    auto threadPool = CreateGrpcThreadPool(conf);
    auto sc = CreateGrpcServiceConnection<AccessService>(
        conf,
        true,
        registry,
        std::move(threadPool),
        std::move(clientId));

    return MakeIntrusive<TGrpcAccessServiceClient>(std::move(sc));
}

} // namespace NSolomon

template <>
void Out<NSolomon::TIamAccount>(IOutputStream& Out, const NSolomon::TIamAccount& acc) {
    switch (acc.Type) {
        case NSolomon::EIamAccountType::User:
            Out << "User {"
                << " id: " << acc.Id << ";"
                << " federation_id: " << (acc.ScopeId.Empty() ? "<empty>" : acc.ScopeId) << ";"
                << " }" << Endl;
            break;
        case NSolomon::EIamAccountType::Service:
            Out << "Service { id: " << acc.Id << "; folder_id: " << acc.ScopeId << "; }" << Endl;
            break;
        case NSolomon::EIamAccountType::Anonymous:
            Out << "Anonymous {}" << Endl;
            break;
    }
}
