#include <solomon/libs/cpp/cloud/access_service/access_service.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/getopt/modchooser.h>

#include <util/string/split.h>

using namespace NLastGetopt;
using namespace NSolomon;

NMonitoring::TMetricRegistry DUMMY_REGISTRY;

class TCommandBase: public TMainClassArgs {
protected:
    virtual void DoRegisterOptions(NLastGetopt::TOpts& opts) = 0;
    virtual int DoExecute(
            NLastGetopt::TOptsParseResult&& opts,
            IAccessServiceClientPtr accessServiceClient) = 0;

private:
    void RegisterOptions(NLastGetopt::TOpts& opts) override {
        opts.AddLongOption('p', "port", "port to connect to")
                .RequiredArgument("PORT")
                .StoreResult(&Port_)
                .DefaultValue(4760);
        opts.AddLongOption('h', "host", "host to connect to")
                .RequiredArgument("HOST")
                .StoreResult(&Host_)
                .DefaultValue("localhost");

        DoRegisterOptions(opts);
    }

    int DoRun(NLastGetopt::TOptsParseResult&& opts) override {
        yandex::solomon::config::rpc::TGrpcClientConfig conf;

        auto* addr = conf.AddAddresses();
        *addr = TStringBuilder() << Host_ << ':' << Port_;

        return DoExecute(
                std::move(opts),
                CreateAccessServiceGrpcClient(conf, DUMMY_REGISTRY)
        );
    }

protected:
    TString Host_;
    ui16 Port_;
};

class TAuthenticateCommand: public TCommandBase {
private:
    void DoRegisterOptions(NLastGetopt::TOpts& opts) override {
        opts.AddLongOption("iam-token")
                .RequiredArgument("STRING");
    }

    int DoExecute(
            NLastGetopt::TOptsParseResult&& opts,
            IAccessServiceClientPtr accessServiceClient) override
    {
        auto token = opts.Get<TString>("iam-token");
        auto f = accessServiceClient->Authenticate(token);

        auto res = f.ExtractValueSync();
        if (!res.Success()) {
            Cerr << "failed to check authentication: " << res.Error().Message << Endl;
            return 1;
        } else {
            Cout << res.Value() << Endl;

            return 0;
        }
    }
};

class TAuthorizeCommand: public TCommandBase {
private:
    void DoRegisterOptions(NLastGetopt::TOpts& opts) override {
        opts.AddLongOption("resources", "In the format \"(type,id),(type,id)\""
                                        ", e.g. \"(resource-manager.folder,yc.monitoring.gateway)\"")
            .RequiredArgument("[]TUPLE");
        opts.AddLongOption("sa_id", "service account id").RequiredArgument("STRING");
        opts.AddLongOption("permission", "permission string id, e.g. monitoring.data.write").RequiredArgument("STRING");
    }

    struct TResource {
        TString Type;
        TString Id;
    };

    TVector<TResource> GetResources(const TString& resourcesStr) {
        TVector<TResource> resources;

        auto withoutLastBrace = TStringBuf{resourcesStr.data(), resourcesStr.size() - 1};
        TVector<TString> tuples = StringSplitter(withoutLastBrace).SplitByString(TStringBuf("),"));

        for (const auto& tuple: tuples) {
            TVector<TString> pair = StringSplitter(tuple).Split(',');
            auto withoutFirstBrace = TString{pair[0].data() + 1, pair[0].size() - 1};

            resources.push_back({
                .Type = withoutFirstBrace,
                .Id = pair[1],
            });
        }

        return resources;
    }

    int DoExecute(
            NLastGetopt::TOptsParseResult&& opts,
            IAccessServiceClientPtr accessServiceClient) override
    {
        auto resources = GetResources(opts.Get<TString>("resources"));
        auto saId = opts.Get<TString>("sa_id");
        auto permission = opts.Get<TString>("permission");

        yandex::cloud::priv::servicecontrol::v1::AuthorizeRequest req;
        req.set_permission(permission);

        auto* subj = req.mutable_subject();
        subj->mutable_service_account()->set_id(saId);
        auto* rp = req.mutable_resource_path();

        for (const auto& resource: resources) {
            auto* r = rp->Add();
            r->set_type(resource.Type);
            r->set_id(resource.Id);
        }

        auto f = accessServiceClient->Authorize(req);

        auto res = f.ExtractValueSync();
        if (!res.Success()) {
            Cerr << "not authorized: " << res.Error().Msg << Endl;
            return 1;
        } else {
            Cout << "OK" << Endl;

            return 0;
        }
    }
};

template <class TCommand>
TMainClassArgs* GetCommand() {
    return Singleton<TCommand>();
}

int Main(int argc, const char** argv) {
    NLastGetopt::NComp::TCustomCompleter::FireCustomCompleter(argc, argv);

    TModChooser modeChooser;
    modeChooser.SetDescription("access_service_client -- a console client for the gRPC AccessService");
    modeChooser.SetPrintShortCommandInUsage(false);
    modeChooser.SetModesHelpOption("-h");

    modeChooser.AddGroupModeDescription("gRPC commands", false);
    modeChooser.AddMode(
            "authenticate",
            GetCommand<TAuthenticateCommand>(),
            "sends an Authenticate request to Access Service",
            false,
            false);
    modeChooser.AddMode(
            "authorize",
            GetCommand<TAuthorizeCommand>(),
            "sends an Authorize request to Access Service",
            false,
            false);

    modeChooser.AddGroupModeDescription("internal operations", false);
    modeChooser.AddCompletions("ingestor_cli", "completion", false, true);

    return modeChooser.Run(argc, argv);
};

int main(int argc, const char** argv) {
    try {
        return Main(argc, argv);
    } catch (...) {
        Cerr << "Unhandled exception: " << CurrentExceptionMessage() << ". Process will terminate" << Endl;
        return 1;
    }
}

