#include "cluster_load_assignment.h"

using envoy::config::endpoint::v3::ClusterLoadAssignment;

namespace NSolomon::NCloud::NEnvoy {

TErrorOr<std::vector<TEndpoint>, TGenericError> GetEndpoints(ClusterLoadAssignment* cluster) {
    size_t size = 0;
    for (const auto& endpoint: cluster->endpoints()) {
        size += endpoint.lb_endpoints_size();
    }

    std::vector<TEndpoint> result;
    result.reserve(size);

    for (auto& endpoint: *cluster->mutable_endpoints()) {
        for (auto& lbEndpoint: *endpoint.mutable_lb_endpoints()) {
            auto* e = lbEndpoint.mutable_endpoint();
            if (!e->has_address() || !e->address().has_socket_address()) {
                continue;
            }

            auto* hostname = e->mutable_hostname();
            if (hostname->empty()) {
                return TGenericError{"unexpected an empty hostname, in cluster: " + cluster->cluster_name()};
            }

            auto* address = e->mutable_address()->mutable_socket_address()->mutable_address();
            if (address->empty()) {
                return TGenericError{"unexpected an empty address, in cluster: " + cluster->cluster_name()};
            }

            result.emplace_back(TEndpoint{std::move(*hostname), std::move(*address)});
        }
    }

    return result;
}

TErrorOr<std::vector<TEndpoint>, TGenericError> ParseClusterLoadAssignment(const google::protobuf::Any& resource) {
    if (auto* arena = resource.GetArena()) {
        auto* cluster = google::protobuf::Arena::CreateMessage<ClusterLoadAssignment>(arena);
        if (!resource.UnpackTo(cluster)) {
            return TGenericError{"cannot parse ClusterLoadAssignment from resource"};
        }
        return GetEndpoints(cluster);
    }

    ClusterLoadAssignment cluster;
    if (!resource.UnpackTo(&cluster)) {
        return TGenericError{"cannot parse ClusterLoadAssignment from resource"};
    }
    return GetEndpoints(&cluster);
}

} // namespace NSolomon::NCloud::NEnvoy
