#include "endpoint_v3_rpc.h"

#include <solomon/libs/cpp/grpc/client/client.h>

#include <cloud/bitbucket/private-api/third_party/envoy/api/envoy/service/endpoint/v3/eds.grpc.pb.h>

using namespace envoy::service::endpoint::v3;
using namespace envoy::service::discovery::v3;

namespace NSolomon::NCloud::NEnvoy {
namespace {

class TEndpointGrpc: public IEndpointRpc {
public:
    explicit TEndpointGrpc(std::unique_ptr<TGrpcServiceConnection<EndpointDiscoveryService>> connection) noexcept
        : Connection_{std::move(connection)}
    {
    }

    TDescoveryAsyncResponse FetchEndpoints(const DiscoveryRequest& req) override {
        return Request<DiscoveryRequest, DiscoveryResponse, &EndpointDiscoveryService::Stub::AsyncFetchEndpoints>(req);
    }

private:
    template <typename TReq, typename TRes, auto Request>
    NThreading::TFuture<TErrorOr<TRes, NGrpc::TGrpcStatus>> Request(const TReq& request) {
        auto promise = NThreading::NewPromise<TErrorOr<TRes, NGrpc::TGrpcStatus>>();
        auto cb = [promise] (NGrpc::TGrpcStatus&& status, TRes&& result) mutable {
            if (!status.Ok()) {
                promise.SetValue(std::move(status));
            } else {
                promise.SetValue(std::move(result));
            }
        };

        Connection_->Request<TReq, TRes>(request, std::move(cb), Request);
        return promise.GetFuture();
    }

private:
    std::unique_ptr<TGrpcServiceConnection<EndpointDiscoveryService>> Connection_;
};

} // namespace

IEndpointRpcPtr CreateEndpointGrpc(
        const yandex::solomon::config::rpc::TGrpcClientConfig& conf,
        NMonitoring::IMetricRegistry& registry,
        TString clientId)
{
    auto threadPool = CreateGrpcThreadPool(conf);
    auto connection = CreateGrpcServiceConnection<EndpointDiscoveryService>(
            conf,
            false,
            registry,
            std::move(threadPool),
            std::move(clientId));

    return std::make_unique<TEndpointGrpc>(std::move(connection));
}

IEndpointClusterRpcPtr CreateEndpointClusterGrpc(
        const yandex::solomon::config::rpc::TGrpcClientConfig& conf,
        NMonitoring::IMetricRegistry& registry,
        TString clientId)
{
    return std::make_shared<TGrpcClusterClientBase<EndpointDiscoveryService, IEndpointRpc>>(
            conf,
            false,
            registry,
            [](auto& registry, auto connection) {
                Y_UNUSED(registry);
                return std::make_unique<TEndpointGrpc>(std::move(connection));
            },
            std::move(clientId));
}

} // namespace NSolomon::NCloud::NEnvoy
