#include <solomon/libs/cpp/cloud/envoy/endpoint_v3_rpc.h>

#include <library/cpp/grpc/server/grpc_request.h>
#include <library/cpp/grpc/server/grpc_server.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/resource/resource.h>
#include <library/cpp/testing/common/network.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <cloud/bitbucket/private-api/third_party/envoy/api/envoy/service/endpoint/v3/eds.grpc.pb.h>
#include <cloud/bitbucket/private-api/third_party/envoy/api/envoy/config/endpoint/v3/endpoint.pb.h>

using namespace envoy::service::endpoint::v3;
using namespace envoy::service::discovery::v3;
using namespace google::protobuf;
using namespace NSolomon::NCloud;

class TEndpointDiscoveryService: public ::NGrpc::TGrpcServiceBase<EndpointDiscoveryService> {
public:
    void InitService(grpc::ServerCompletionQueue* cq, ::NGrpc::TLoggerPtr logger) override {
        MakeIntrusive<::NGrpc::TGRpcRequest<DiscoveryRequest, DiscoveryResponse, TEndpointDiscoveryService>>(
                this,
                &Service_,
                cq,
                [](::NGrpc::IRequestContextBase* ctx) {
                    auto pbText = NResource::Find("/test_cluster_assignment.txt");
                    envoy::config::endpoint::v3::ClusterLoadAssignment cluster;
                    Y_ENSURE(TextFormat::ParseFromString(pbText, &cluster));

                    DiscoveryResponse resp;
                    auto* resource = resp.add_resources();
                    Y_ENSURE(resource->PackFrom(cluster));

                    ctx->Reply(&resp);
                },
                &EndpointDiscoveryService::AsyncService::RequestFetchEndpoints,
                "FetchEndpoints",
                logger,
                ::NGrpc::FakeCounterBlock())->Run();
    }

    bool IncRequest() {
        return true;
    }

    void DecRequest() {
    }
};

TEST(TEnvoyEndpointTest, FetchEndpoints) {
    auto port = NTesting::GetFreePort();

    // (1) start mock server
    auto server = MakeHolder<::NGrpc::TGRpcServer>(::NGrpc::TServerOptions().SetHost("localhost").SetPort(port));
    server->AddService(MakeIntrusive<TEndpointDiscoveryService>());
    server->Start();

    // (2) create client
    NMonitoring::TMetricRegistry metrics;
    yandex::solomon::config::rpc::TGrpcClientConfig config;
    config.add_addresses(TStringBuilder{} << "localhost:" << port);
    auto rpc = NEnvoy::CreateEndpointGrpc(config, metrics, "unit-test");

    // (3) send request
    envoy::service::discovery::v3::DiscoveryRequest req;
    req.add_resource_names("ig.cloud.yandex.net.");
    if (auto* node = req.mutable_node()) {
        node->set_id("test");
        node->set_user_agent_name("solomon");
        node->set_user_agent_version("cloud/v2");
    }
    auto resp = rpc->FetchEndpoints(req).ExtractValueSync();

    // (4) check response
    ASSERT_TRUE(resp.Success()) << "code: " << resp.Error().GRpcStatusCode << ", message: " << resp.Error().Msg;
    ASSERT_EQ(resp.Value().resources_size(), 1);

    for (const auto& resource: resp.Value().resources()) {
        envoy::config::endpoint::v3::ClusterLoadAssignment value;
        ASSERT_TRUE(resource.UnpackTo(&value));
        // Cout << value.DebugString() << Endl;

        ASSERT_GT(value.endpoints_size(), 0);

        for (const auto& endpoint: value.endpoints()) {
            ASSERT_GT(endpoint.lb_endpoints_size(), 0);

            for (const auto& lbEndpoint: endpoint.lb_endpoints()) {
                ASSERT_TRUE(lbEndpoint.has_endpoint());

                const auto& e = lbEndpoint.endpoint();
                ASSERT_TRUE(e.has_address());
                ASSERT_TRUE(e.address().has_socket_address());

                const auto& hostname = e.hostname();
                ASSERT_TRUE(!hostname.empty());

                const auto& address = e.address().socket_address().address();
                ASSERT_TRUE(!address.empty());

                Cout << hostname << " => " << address << Endl;
            }
        }
    }

    server->Stop();
}
