#include "client.h"

#include <solomon/libs/cpp/grpc/client/client.h>
#include <solomon/libs/cpp/grpc/status/code.h>

#include <cloud/bitbucket/private-api/yandex/cloud/priv/iam/v1/iam_token_service.pb.h>
#include <cloud/bitbucket/private-api/yandex/cloud/priv/iam/v1/iam_token_service.grpc.pb.h>

using namespace NMonitoring;
using namespace NThreading;
using namespace yandex::cloud::priv::iam::v1;

namespace NRpcConf = yandex::solomon::config::rpc;

namespace NSolomon::NCloud {
namespace {
    struct TIamToken: IToken {
    public:
        explicit TIamToken(TString token, TInstant expiresAt)
            : Token_{std::move(token)}
            , ExpiresAt_{expiresAt}
        {
        }

        TStringBuf Token() const override {
            return Token_;
        }

        TInstant ExpiresAt() const override {
            return ExpiresAt_;
        }

    private:
        TString Token_;
        TInstant ExpiresAt_;
    };

    class TGrpcClient final: public IIamClient {
    public:
        explicit TGrpcClient(std::unique_ptr<TGrpcServiceConnection<IamTokenService>> connection)
            : Connection_{std::move(connection)}
        {
        }

        NThreading::TFuture<ITokenPtr> CreateIamToken(TString jwt) override {
            CreateIamTokenRequest req;
            req.set_jwt(jwt);

            auto promise = NewPromise<ITokenPtr>();

            auto cb = [promise] (NGrpc::TGrpcStatus&& status, CreateIamTokenResponse&& result) mutable {
                if (!status.Ok()) {
                    TStringBuilder msg;
                    msg << "status: " << NGrpc::StatusCodeToString(status.GRpcStatusCode);
                    if (!status.Msg.empty()) {
                        msg << ", message: " << status.Msg;
                    }
                    promise.SetException(msg);
                } else {
                    promise.SetValue(MakeIntrusive<TIamToken>(
                        result.iam_token(),
                        TInstant::Seconds(result.expires_at().seconds())
                    ));
                }
            };

            Connection_->Request<CreateIamTokenRequest, CreateIamTokenResponse>(
                std::move(req),
                std::move(cb),
                &IamTokenService::Stub::AsyncCreate);

            return promise;
        }

    private:
        std::unique_ptr<TGrpcServiceConnection<IamTokenService>> Connection_;
    };
} // namespace

IIamClientPtr CreateGrpcIamClient(const NRpcConf::TGrpcClientConfig& conf, IMetricRegistry& registry, TString clientId) {
    auto threadPool = CreateGrpcThreadPool(conf);
    auto sc = CreateGrpcServiceConnection<IamTokenService>(
        conf,
        true,
        registry,
        std::move(threadPool),
        std::move(clientId));

    return MakeIntrusive<TGrpcClient>(std::move(sc));
}

} // namespace NSolomon::NCloud
