#include "iam.h"
#include "client.h"

#include <solomon/libs/cpp/http/client/curl/client.h>
#include <solomon/libs/cpp/steady_timer/steady_timer.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/monlib/metrics/metric.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/threading/hot_swap/hot_swap.h>

#include <util/generic/scope.h>
#include <util/stream/file.h>
#include <util/string/builder.h>
#include <util/string/strip.h>
#include <util/system/env.h>

#include <chrono>
#include <utility>

#include <contrib/libs/jwt-cpp/include/jwt-cpp/jwt.h>

namespace NSolomon::NCloud {
namespace {
    using namespace NThreading;

    constexpr auto EXPIRATION_TIME_MIN = TDuration::Minutes(10);
    constexpr auto PING_INTERVAL = TDuration::Minutes(1);

    class TProviderBase {
    public:
        TProviderBase(TTokenProviderConf conf, IIamClientPtr iamClient)
            : Config_{std::move(conf)}
            , IamClient_{std::move(iamClient)}
        {
        }

        TString CreateJwt() {
            static const TString AUDIENCE{"https://iam.api.cloud.yandex.net/iam/v1/tokens"};

            std::string serviceAccountId{Config_.ServiceAccountId};
            std::string keyId{Config_.KeyId};

            const auto now = std::chrono::system_clock::now();
            const auto expire = now + std::chrono::milliseconds(Config_.TokenLifetime.MilliSeconds());

            const TString pubKey = Config_.PublicKey ?  Config_.PublicKey : TFileInput{Config_.PublicKeyFile}.ReadAll();
            const TString privKey = Config_.PrivateKey ? Config_.PrivateKey : TFileInput{Config_.PrivateKeyFile}.ReadAll();

            const auto token = jwt::create()
                .set_key_id(keyId)
                .set_issuer(serviceAccountId)
                .set_issued_at(now)
                .set_audience(AUDIENCE)
                .set_expires_at(expire)
                .sign(jwt::algorithm::ps256(pubKey, privKey));

            return TString{token};
        }

    protected:
        TTokenProviderConf Config_;
        IIamClientPtr IamClient_;
    };

    bool ShouldRenewToken(const ITokenPtr& token, TInstant now, TInstant lastSuccess, TDuration updatePeriod) {
        if (!token) {
            return true;
        }

        TInstant expiresAt = token->ExpiresAt();

        if (now >= expiresAt) {
            return true;
        }

        return (now - lastSuccess) >= updatePeriod
            || (expiresAt - now) <= EXPIRATION_TIME_MIN;
    }

    class TTokenProvider final: public TProviderBase, public ITokenProvider {
    public:
        TTokenProvider(TTokenProviderConf conf, const IIamClientPtr& iamClient, ILoggerPtr logger)
            : TProviderBase{std::move(conf), iamClient}
            , Logger_{std::move(logger)}
        {
            Token_ = iamClient->CreateIamToken(CreateJwt()).GetValueSync();
        }

        ITokenPtr Token() const override {
            return Token_;
        }

        void Start(EStartPolicy) override {}

    private:
        ITokenPtr Token_;
        ILoggerPtr Logger_;
    };

    class TUpdatingProvider final: public TProviderBase, public ITokenProvider {
    public:
        TUpdatingProvider(const TUpdatingProviderConf& conf, IExecutor& executor, IIamClientPtr iamClient, ICountersPtr counters, ILoggerPtr logger)
            : TProviderBase{conf, std::move(iamClient)}
            , UpdatePeriod_{conf.UpdatePeriod}
            , Executor_{executor}
            , Counters_{std::move(counters)}
            , Logger_{std::move(logger)}
        {
        }

        ~TUpdatingProvider() {
            Executor_.Stop();
            if (Outstanding_.Initialized()) {
                Outstanding_.Wait();
            }
        }

        void Start(EStartPolicy startPolicy) override {
            PingSelf(startPolicy);

            Executor_.Schedule([self{TIntrusivePtr{this}}] {
                self->PingSelf(EStartPolicy::Async);
            }, PING_INTERVAL, PING_INTERVAL);
        }

    private:
        ITokenPtr Token() const override {
            return Token_.AtomicLoad();
        }

        void PingSelf(EStartPolicy startPolicy) {
            const auto now = TInstant::Now();
            const auto lastSuccess = LastSuccess();
            const auto token = Token();

            if (ShouldRenewToken(token, now, lastSuccess, UpdatePeriod_)) {
                DoRenewToken();
                switch (startPolicy) {
                    case EStartPolicy::Sync: {
                        if (Outstanding_.Initialized()) {
                            Outstanding_.Wait(PING_INTERVAL);
                        }
                        break;
                    }
                    case EStartPolicy::Async:
                        // do not block
                        break;
                }
            }

            Counters_->SetTokenAge(now - lastSuccess);

            if (token) {
                const auto expiresAt = token->ExpiresAt();
                const auto remainingLifetime = now >= expiresAt ? TDuration::Zero() : (expiresAt - now);

                Counters_->SetRemainingLifetime(remainingLifetime);
            }
        }

        void HandleIamResponse(const TFuture<ITokenPtr>& f) noexcept {
            try {
                Token_.AtomicStore(f.GetValue());
                Counters_->Success();
                Counters_->SetTokenAge(TDuration::Zero());
                LastSuccess_.store(TInstant::Now().MilliSeconds());
            } catch (...) {
                Logger_->Write(
                        ELogPriority::TLOG_ERR,
                        TStringBuilder{} << "cannot handle IAM response: " << CurrentExceptionMessage());
                Counters_->Fail();
            }
        }

        void DoRenewToken() noexcept {
            try {
                Outstanding_ = IamClient_->CreateIamToken(CreateJwt())
                        .Apply([self{TIntrusivePtr{this}}](auto&& f) {
                            self->HandleIamResponse(f);
                        });
            } catch (...) {
                Logger_->Write(
                        ELogPriority::TLOG_ERR,
                        TStringBuilder{} << "cannot renew IAM token: " << CurrentExceptionMessage());
            }
        }

        TInstant LastSuccess() const {
            return TInstant::MilliSeconds(LastSuccess_.load());
        }

    private:
        TDuration UpdatePeriod_;
        IExecutor& Executor_;
        ICountersPtr Counters_;
        ILoggerPtr Logger_;
        TFuture<void> Outstanding_;

        // these two can be changed not atomically, but this is fine
        THotSwap<IToken> Token_;
        std::atomic<ui64> LastSuccess_{0};
    };

    struct TAccessToken: IToken {
    public:
        explicit TAccessToken(TString token, TInstant expiresAt)
            : Token_{std::move(token)}
            , ExpiresAt_{expiresAt}
        {
        }

        TStringBuf Token() const override {
            return Token_;
        }

        TInstant ExpiresAt() const override {
            return ExpiresAt_;
        }
    private:
        TString Token_;
        TInstant ExpiresAt_;
    };

    struct TTokenProviderMetrics {
        NMonitoring::TIntGauge* TokenAgeSeconds;
        NMonitoring::TIntGauge* RemainingLifetimeSeconds;
        NMonitoring::TRate* RequestOk;
        NMonitoring::TRate* RequestError;
        NMonitoring::THistogram* ResponseTimeMs;
    };

    class TMetadataServiceTokenProvider final: public ITokenProvider {
    public:
        TMetadataServiceTokenProvider(
                TMetadataServiceProviderConf conf,
                THolder<IExecutor> executor,
                NMonitoring::TMetricRegistry& registry,
                ILoggerPtr logger)
            : Conf_{std::move(conf)}
            , Executor_{std::move(executor)}
            , Registry_{registry}
            , Logger_{std::move(logger)}
        {
            TCurlClientOptions curlOpts;
            curlOpts.DnsCacheLifetime = TDuration::Minutes(5);
            curlOpts.MaxInflight = 1;
            curlOpts.QueueSizeLimit = 1;

            if (TString caCertDir = Strip(GetEnv("SA_CAPATH", "")); !caCertDir.empty()) {
                curlOpts.CaCertDir = caCertDir;
            }

            HttpClient_ = CreateCurlClient(curlOpts, Registry_);

            Metrics_.RemainingLifetimeSeconds = Registry_.IntGauge({{"sensor", "iamMetadata.tokenRemainingSeconds"}});
            Metrics_.TokenAgeSeconds = Registry_.IntGauge({{"sensor", "iamMetadata.tokenAgeSeconds"}});
            Metrics_.RequestOk = Registry_.Rate({{"sensor", "iamMetadata.requestOk"}});
            Metrics_.RequestError = Registry_.Rate({{"sensor", "iamMetadata.requestError"}});
            Metrics_.ResponseTimeMs = Registry_.HistogramRate(
                    {
                            {"sensor", "iamMetadata.responseTimeMs"},
                    },
                    NMonitoring::ExponentialHistogram(13, 2, 16));
        }

        ~TMetadataServiceTokenProvider() override {
            Executor_->Stop();
            if (Outstanding_.Initialized()) {
                Outstanding_.Wait();
            }
        }

        void Start(EStartPolicy startPolicy) override {
            PingSelf(startPolicy);

            Executor_->Schedule([self{TIntrusivePtr{this}}] {
                self->PingSelf(EStartPolicy::Async);
            }, PING_INTERVAL, PING_INTERVAL);
        }

        ITokenPtr Token() const override {
            return Token_.AtomicLoad();
        }

    private:
        void PingSelf(EStartPolicy startPolicy) {
            const auto now = TInstant::Now();
            const auto lastSuccess = LastSuccess();
            const auto token = Token();

            if (ShouldRenewToken(token, now, lastSuccess, Conf_.UpdatePeriod)) {
                DoRenewToken();
                switch (startPolicy) {
                    case EStartPolicy::Sync: {
                        if (Outstanding_.Initialized()) {
                            Outstanding_.Wait(PING_INTERVAL);
                        }
                        break;
                    }
                    case EStartPolicy::Async:
                       // don't block anything
                       break;
                }
            }

            Metrics_.TokenAgeSeconds->Set((now - lastSuccess).Seconds());

            if (token) {
                const auto expiresAt = token->ExpiresAt();
                const auto remainingLifetime = now >= expiresAt ? TDuration::Zero() : (expiresAt - now);

                Metrics_.RemainingLifetimeSeconds->Set(remainingLifetime.Seconds());
            }
        }

        void OnRequestCompleted(bool failed, TDuration respTime) const {
            if (failed) {
                Metrics_.RequestError->Inc();
            } else {
                Metrics_.RequestOk->Inc();
            }

            Metrics_.ResponseTimeMs->Record(respTime.MilliSeconds());
        }

        auto CreateMetadataResponseHandler(const TString& url) {
            return [
                    self{TIntrusivePtr{this}},
                    logger{Logger_},
                    timer{TSteadyTimer()},
                    url
            ](IHttpClient::TResult result) mutable {
                TDuration respTime = timer.Step();
                bool failed = false;

                Y_DEFER {
                     self->OnRequestCompleted(failed, respTime);
                };

                if (!result.Success()) {
                    failed = true;
                    logger->Write(
                            ELogPriority::TLOG_WARNING,
                            TStringBuilder() << "failed to request data from " << url
                            << ": " << result.Error().Message());
                } else if (result.Value()->Code() >= 400) {
                    failed = true;
                    logger->Write(
                            ELogPriority::TLOG_WARNING,
                            TStringBuilder() << "failed to request data from " << url << " with code "
                                << result.Value()->Code() << ": " << result.Value()->Data());
                } else {
                    logger->Write(
                            ELogPriority::TLOG_DEBUG,
                            TStringBuilder() << "got a response from " << url);

                    if (result.Fail()) {
                        logger->Write(
                                ELogPriority::TLOG_WARNING,
                                TStringBuilder() << "got an error while requesting metadata: "
                                    << result.Error().Message());
                    } else {
                        TString tokenStr;
                        TInstant expiresAt;

                        NJson::TJsonValue json;
                        NJson::TJsonReaderConfig readerConfig;
                        readerConfig.DontValidateUtf8 = true;

                        auto valuePtr = std::move(result.Value());
                        TStringBuf jsonKey;

                        try {
                            NJson::ReadJsonTree(valuePtr->Data(), &readerConfig, &json, true);

                            jsonKey = TStringBuf("expires_in");
                            expiresAt = TInstant::Now()
                                + TDuration::Seconds(json[jsonKey].GetUIntegerSafe())
                                - TDuration::Seconds(1);

                            jsonKey = TStringBuf("access_token");
                            tokenStr = json[jsonKey].GetStringSafe();
                        } catch (...) {
                            failed = true;

                            auto errMsg = TStringBuilder() << "failed to parse a response"
                                << " from the metadata service: ";

                            if (!jsonKey.empty()) {
                                errMsg += "(key=";
                                errMsg += jsonKey;
                                errMsg += ") ";
                            }

                            errMsg += CurrentExceptionMessage();

                            logger->Write(ELogPriority::TLOG_WARNING, errMsg);
                            return;
                        }

                        ITokenPtr tokenPtr = MakeIntrusive<TAccessToken>(tokenStr, expiresAt);
                        self->Token_.AtomicStore(std::move(tokenPtr));
                    }
                }
            };
        }

        void DoRenewToken() {
            TRequestOpts opts;
            opts.ConnectTimeout = TDuration::Seconds(5);
            opts.ReadTimeout = TDuration::Minutes(5);
            opts.Retries = 3;
            opts.BackoffMin = TDuration::Seconds(5);
            opts.BackoffMax = TDuration::Minutes(1);

            auto url = Conf_.MetadataServiceAddress;
            auto headers = Headers({{"Metadata-Flavor", "Google"}});
            auto req = Get(url, std::move(headers));

            HttpClient_->Request(std::move(req), CreateMetadataResponseHandler(url), opts);
        }

        TInstant LastSuccess() const {
            return TInstant::MilliSeconds(LastSuccess_.load());
        }

    private:
        TMetadataServiceProviderConf Conf_;
        THolder<IExecutor> Executor_;
        NMonitoring::TMetricRegistry& Registry_;
        ILoggerPtr Logger_;
        TFuture<void> Outstanding_;
        IHttpClientPtr HttpClient_;

        // these two can be changed not atomically, but this is fine
        THotSwap<IToken> Token_;
        std::atomic<ui64> LastSuccess_{0};
        TTokenProviderMetrics Metrics_;
    };

    class TStaticTokenProvider: public ITokenProvider {
    public:
        TStaticTokenProvider(TString token)
            : Token_{std::move(token)}
        {
        }

        ITokenPtr Token() const override {
            return new TAccessToken{Token_, TDuration::Minutes(5).ToDeadLine() };
        }

        void Start(EStartPolicy) override {}

    private:
        TString Token_;
    };
}

    ITokenProviderPtr CreateUpdatingProvider(
        TUpdatingProviderConf conf,  // NOLINT(performance-unnecessary-value-param): false positive
        IExecutor& executor,
        IIamClientPtr iamClient,
        ICountersPtr counters,
        ILoggerPtr logger)
    {
        return new TUpdatingProvider(std::move(conf), executor, std::move(iamClient), std::move(counters), std::move(logger));
    }

    // NOLINTNEXTLINE(performance-unnecessary-value-param): false positive
    ITokenProviderPtr CreateTokenProvider(TTokenProviderConf conf, IIamClientPtr iamClient, ILoggerPtr logger) {
        return new TTokenProvider{std::move(conf), std::move(iamClient), std::move(logger)};
    }

    ITokenProviderPtr CreateMetadataServiceTokenProvider(
            TMetadataServiceProviderConf conf,
            THolder<IExecutor> executor,
            NMonitoring::TMetricRegistry& registry,
            ILoggerPtr logger)
    {
        return new TMetadataServiceTokenProvider{
            std::move(conf),
            std::move(executor),
            registry,
            std::move(logger),
        };
    }

    ITokenProviderPtr CreateStaticTokenProvider(TString token) {
        return new TStaticTokenProvider(std::move(token));
    }

} // namespace NSolomon::NCloud
