#pragma once

#include <library/cpp/logger/priority.h>
#include <library/cpp/monlib/metrics/fwd.h>

#include <util/datetime/base.h>
#include <util/folder/path.h>
#include <util/generic/ptr.h>
#include <util/generic/strbuf.h>

namespace NSolomon::NCloud {
    struct IToken: TThrRefBase {
        virtual TStringBuf Token() const = 0;
        virtual TInstant ExpiresAt() const = 0;
    };

    using ITokenPtr = TIntrusivePtr<IToken>;

    struct IExecutor {
        virtual ~IExecutor() = default;
        virtual void Schedule(std::function<void(void)> fn, TDuration delay, TDuration interval) = 0;
        virtual void Stop() = 0;
    };

    enum class EStartPolicy {
        /**
         * Block until the first token value is gathered
         */
        Sync = 0,
        /**
         * Start and don't block anything
         */
        Async,
    };

    struct ITokenProvider: TThrRefBase {
    public:
        virtual ITokenPtr Token() const = 0;
        virtual void Start(EStartPolicy startPolicy = EStartPolicy::Async) = 0;
    };

    using ITokenProviderPtr = TIntrusivePtr<ITokenProvider>;

    struct TTokenProviderConf {
        TString ServiceAccountId;
        TString KeyId;

        TFsPath PublicKeyFile;
        TFsPath PrivateKeyFile;
        TString PublicKey;
        TString PrivateKey;

        TDuration TokenLifetime{TDuration::Hours(1)};
    };

    struct TUpdatingProviderConf: TTokenProviderConf {
        TDuration UpdatePeriod{TDuration::Hours(45)};
    };

    struct TMetadataServiceProviderConf {
        TString MetadataServiceAddress;
        TDuration UpdatePeriod{TDuration::Hours(45)};
    };

    struct ICounters {
        virtual ~ICounters() = default;
        virtual void Fail() = 0;
        virtual void Success() = 0;
        virtual void SetTokenAge(TDuration) = 0;
        virtual void SetRemainingLifetime(TDuration) = 0;
    };

    struct ILogger: TThrRefBase {
        virtual void Write(ELogPriority severity, TString msg) = 0;
    };

    using ICountersPtr = THolder<ICounters>;
    using ILoggerPtr = TIntrusivePtr<ILogger>;

    ITokenProviderPtr CreateUpdatingProvider(
        TUpdatingProviderConf conf,
        IExecutor& executor,
        TIntrusivePtr<class IIamClient> iamClient,
        ICountersPtr counters,
        ILoggerPtr logger);

    ITokenProviderPtr CreateMetadataServiceTokenProvider(
        TMetadataServiceProviderConf conf,
        THolder<IExecutor> executor,
        class NMonitoring::TMetricRegistry& registry,
        ILoggerPtr logger);

    // get token only once
    ITokenProviderPtr CreateTokenProvider(
        TTokenProviderConf conf,
        TIntrusivePtr<class IIamClient> iamClient,
        ILoggerPtr logger
    );

    // will provide static token and never update it
    ITokenProviderPtr CreateStaticTokenProvider(TString token);

} // namespace NSolomon::NCloud
