#include <solomon/libs/cpp/cloud/iam/iam.h>
#include <solomon/libs/cpp/cloud/iam/client.h>
#include <solomon/libs/cpp/cloud/instance_group/client.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

using namespace NSolomon::NCloud;
using namespace yandex::solomon::config::rpc;
using namespace NLastGetopt;

struct TLogger: ILogger {
    void Write(ELogPriority, TString msg) override {
        Cerr << msg << Endl;
    }
};

int main(int argc, const char** argv) {
    TString env, privKey, pubKey, keyId, accId, groupName, folderId;

    TOpts opts = NLastGetopt::TOpts::Default();
    opts.AddHelpOption();
    opts.AddLongOption("env").RequiredArgument("{prod|pre}").StoreResult(&env).DefaultValue("prod").Optional();
    opts.AddLongOption("priv").RequiredArgument("FILE").StoreResult(&privKey).Required();
    opts.AddLongOption("pub").RequiredArgument("FILE").StoreResult(&pubKey).Required();
    opts.AddLongOption('k', "key-id").RequiredArgument("ID").StoreResult(&keyId).Required();
    opts.AddLongOption('a', "acc-id").RequiredArgument("ID").StoreResult(&accId).Required();
    opts.AddLongOption('g', "instance-group").RequiredArgument("ID").StoreResult(&groupName);
    opts.AddLongOption('f', "folder-id").RequiredArgument("ID").StoreResult(&folderId);
    opts.SetFreeArgsNum(0);

    TOptsParseResult res(&opts, argc, argv);

    if (groupName.empty() == folderId.empty()) {
        Cerr << "Error: only one of folder-id and instance-group must be specified" << Endl;
        return -1;
    }

    NMonitoring::TMetricRegistry registry;

    TGrpcClientConfig grpcConf;
    grpcConf.AddAddresses((env == "prod")
            ? "ts.private-api.cloud.yandex.net:4282"
            : "ts.private-api.cloud-preprod.yandex.net:4282");
    auto iamClient = CreateGrpcIamClient(grpcConf, registry);

    TTokenProviderConf conf {
            .ServiceAccountId = accId,
            .KeyId = keyId,
            .PublicKeyFile = pubKey,
            .PrivateKeyFile = privKey,
    };
    auto tokenProvider = CreateTokenProvider(std::move(conf), iamClient, MakeIntrusive<TLogger>());
    tokenProvider->Start(EStartPolicy::Sync);

    TGrpcClientConfig igGrpcConf;
    igGrpcConf.AddAddresses((env == "prod")
            ? "instance-group.private-api.ycp.cloud.yandex.net"
            : "instance-group.private-api.ycp.cloud-preprod.yandex.net");
    auto instanceGroupClient = CreateInstanceGroupClient(igGrpcConf, tokenProvider, registry);

    TListInstancesResult reply;
    if (!groupName.empty()) {
        reply = instanceGroupClient->ListInstances(groupName)
            .ExtractValueSync();
    } else {
        reply = instanceGroupClient->ListInstancesForFolder(folderId)
            .ExtractValueSync();
    }

    if (!reply.Success()) {
        Cerr << "Error: " << reply.Error().Message() << Endl;
        return -1;
    }

    for (auto&& instance: reply.Value().Instances) {
        Cout << instance.Fqdn << " " << instance.Address << Endl;
    }
}
