#include "client.h"

#include <solomon/libs/cpp/grpc/metrics/counters.h>
#include <solomon/libs/cpp/grpc/client/client.h>

#include <solomon/libs/cpp/config/units.h>

#include <cloud/bitbucket/private-api/yandex/cloud/priv/microcosm/instancegroup/v1/instance_group_service.grpc.pb.h>

#include <library/cpp/threading/future/future.h>
#include <library/cpp/ipmath/ipmath.h>

using namespace NMonitoring;
using namespace NGrpc;
using namespace NThreading;


namespace NCloud = yandex::cloud::priv::microcosm::instancegroup::v1;
using TGrpcConfig = yandex::solomon::config::rpc::TGrpcClientConfig;


namespace NSolomon::NCloud {
namespace {
    const TIpAddressRange GLOBAL_ADDRESSES = TIpAddressRange::FromCidrString("2000::/3");

    TInstanceGroup InstanceGroupFromProto(::NCloud::ListInstanceGroupInstancesResponse&& proto) {
        TInstanceGroup result;
        for (auto&& instance: proto.instances()) {
            if (instance.status() == ::NCloud::ManagedInstance::RUNNING
                || instance.status() == ::NCloud::ManagedInstance::RUNNING_ACTUAL
                || instance.status() == ::NCloud::ManagedInstance::RUNNING_OUTDATED)
            {
                TIpv6Address v6Address;

                for (auto&& intf: instance.network_interfaces()) {
                    const auto addr = intf.primary_v6_address().address();

                    if (addr.empty()) {
                        continue;
                    }

                    bool ok{false};
                    const auto ipAddr = TIpv6Address::FromString(addr, ok);
                    if (!ok) {
                        continue;
                    }

                    if (GLOBAL_ADDRESSES.Contains(ipAddr)) {
                        continue;
                    }

                    v6Address = ipAddr;
                }

                result.Instances.emplace_back(instance.fqdn(), v6Address);
            }
        }

        return result;
    }

    class TInstanceGroupClient final: public IInstanceGroupClient {
    public:
        TInstanceGroupClient(
                std::unique_ptr<TGrpcServiceConnection<::NCloud::InstanceGroupService>> connection,
                ITokenProviderPtr tokenProvider)
            : Connection_{std::move(connection)}
            , TokenProvider_{std::move(tokenProvider)}
        {
        }

        TFuture<TListInstancesResult> ListInstances(TString id) override {
            auto token = TokenProvider_->Token();
            if (!token) {
                return MakeFuture<TListInstancesResult>(TInstanceGroup{});
            }

            ::NCloud::ListInstanceGroupInstancesRequest req;
            req.set_instance_group_id(std::move(id));

            auto promise = NewPromise<TListInstancesResult>();

            TCallMeta meta {
                .CallCredentials = grpc::AccessTokenCredentials(static_cast<std::string>(token->Token())),
                .Timeout = Connection_->Timeout(),
            };

            auto cb = [promise] (TGrpcStatus&& status, ::NCloud::ListInstanceGroupInstancesResponse&& result) mutable {
                if (!status.Ok()) {
                    promise.SetValue(TListInstancesResult::FromError(std::move(status.Msg)));
                } else {
                    promise.SetValue(InstanceGroupFromProto(std::move(result)));
                }
            };

            Connection_->Request<::NCloud::ListInstanceGroupInstancesRequest, ::NCloud::ListInstanceGroupInstancesResponse>(
                std::move(req),
                std::move(cb),
                &::NCloud::InstanceGroupService::Stub::AsyncListInstances,
                std::move(meta)
            );

            return promise;
        }

        TFuture<TListInstancesResult> ListInstancesForFolder(TString folderId) override;
        TFuture<TListInstanceGroupsResult> ListInstanceGroups(TString folderId) override;

        template <typename TCallback>
        void MakeListGroupsRequest(::NCloud::ListInstanceGroupsRequest req, TCallback callback) {
            auto token = TokenProvider_->Token();

            TCallMeta meta {
                .CallCredentials = grpc::AccessTokenCredentials(static_cast<std::string>(token->Token())),
                .Timeout = Connection_->Timeout(),
            };

            Connection_->Request<::NCloud::ListInstanceGroupsRequest, ::NCloud::ListInstanceGroupsResponse>(
                std::move(req),
                std::move(callback),
                &::NCloud::InstanceGroupService::Stub::AsyncList,
                std::move(meta)
            );
        }

    private:
        std::unique_ptr<TGrpcServiceConnection<::NCloud::InstanceGroupService>> Connection_;
        ITokenProviderPtr TokenProvider_;
    };

    class TMultipleRequestContext {
    public:
        // NOLINTNEXTLINE(performance-unnecessary-value-param): false positive
        static TFuture<TListInstancesResult> DoRequest(TVector<TString> groups, IInstanceGroupClientPtr client) {
            if (groups.empty()) {
                return MakeFuture<TListInstancesResult>(TListInstancesResult::FromValue(TInstanceGroup{}));
            }

            auto* ctx = new TMultipleRequestContext{std::move(groups), std::move(client)};
            return ctx->Promise_.GetFuture();
        }

    private:
        TMultipleRequestContext(const TVector<TString>& groups, const IInstanceGroupClientPtr& client) {
            InFlight_ = groups.size();
            Errors_.resize(groups.size());

            size_t idx = 0;
            for (auto&& group: groups) {
                client->ListInstances(group).Subscribe([this, group, idx] (auto f) {
                    CompleteOne(std::move(f), std::move(group), idx);
                });

                ++idx;
            }
        }

        void Destroy() {
            delete this;
        }

        void CompleteOne(TFuture<TListInstancesResult> r, const TString& group, size_t idx) try {
            auto result = r.ExtractValue();
            if (result.Success()) {
                auto part = result.Extract();
                Copy(std::make_move_iterator(part.Instances.begin()),
                    std::make_move_iterator(part.Instances.end()),
                    std::back_inserter(Result_.Instances));
            } else {
                ReportError(TStringBuilder() << group << ": " << result.Error().Message(), idx);
            }

            if (InFlight_.fetch_sub(1, std::memory_order_relaxed) == 1) {
                CompleteSuccess();
            }
        } catch (...) {
            ReportError(TStringBuilder() << group << ": " << CurrentExceptionMessage(), idx);
        }

        void ReportError(TString msg, size_t idx) {
            HasError_ = true;
            Errors_[idx] = std::move(msg);
        }

        void Complete() {
            if (HasError_) {
                CompleteError();
            } else {
                CompleteSuccess();
            }
        }

        void CompleteSuccess() {
            Promise_.SetValue(TListInstancesResult::FromValue(std::move(Result_)));
            Destroy();
        }

        void CompleteError() {
            TStringBuilder combinedError;

            for (auto&& msg: Errors_) {
                combinedError << msg << ";\t";
            }

            Promise_.SetValue(TListInstancesResult::FromError(TString{combinedError}));
            Destroy();
        }

    private:
        std::atomic<i32> InFlight_{0};
        std::atomic<bool> HasError_{false};
        TInstanceGroup Result_;
        TVector<TString> Errors_;
        TPromise<TListInstancesResult> Promise_{NewPromise<TListInstancesResult>()};
    };


    class TFolderListRequestContext {
    public:
        static TFuture<TListInstanceGroupsResult> DoRequest(TString folderId, TIntrusivePtr<TInstanceGroupClient> client) {
            auto* ctx = new TFolderListRequestContext(std::move(folderId), std::move(client));
            return ctx->Promise_.GetFuture();
        }

    private:
        static constexpr auto DEFAULT_PAGE_SIZE = 1000;

        TFolderListRequestContext(TString folderId, TIntrusivePtr<TInstanceGroupClient> client)
            : FolderId_{std::move(folderId)}
            , Client_{std::move(client)}
        {
            MakeRequest();
        }

        ~TFolderListRequestContext() = default;

        void Destroy() {
            delete this;
        }

        void MakeRequest(TString nextPage = {}) {
            ::NCloud::ListInstanceGroupsRequest req;
            req.set_folder_id(FolderId_);
            req.set_page_size(DEFAULT_PAGE_SIZE);
            req.set_view(::NCloud::BASIC);
            req.set_page_token(nextPage);

            Client_->MakeListGroupsRequest(std::move(req), [this] (auto status, auto resp) {
                OnResponse(std::move(status), std::move(resp));
            });
        }

        void CompleteError(TString msg) {
            Promise_.SetValue(TListInstanceGroupsResult::FromError(msg));
            Destroy();
        }

        void CompleteSuccess() {
            Promise_.SetValue(TListInstanceGroupsResult::FromValue(std::move(InstanceGroups_)));
            Destroy();
        }

        void OnResponse(const TGrpcStatus& status, const ::NCloud::ListInstanceGroupsResponse& resp) {
            if (!status.Ok()) {
                CompleteError(status.Msg);
                return;
            }

            for (auto&& ig: resp.instance_groups()) {
                InstanceGroups_.push_back(ig.id());
            }

            const bool hasMore = !resp.next_page_token().empty();

            if (!hasMore) {
                CompleteSuccess();
            } else {
                MakeRequest(resp.next_page_token());
            }
        }

    private:
        TString FolderId_;
        TPromise<TListInstanceGroupsResult> Promise_{NewPromise<TListInstanceGroupsResult>()};
        TIntrusivePtr<TInstanceGroupClient> Client_;
        TVector<TString> InstanceGroups_;
    };

    TFuture<TListInstanceGroupsResult> TInstanceGroupClient::ListInstanceGroups(TString folderId) {
        auto token = TokenProvider_->Token();
        if (!token) {
            return MakeFuture<TListInstanceGroupsResult>(TListInstanceGroupsResult::FromError("IAM token is empty"));
        }

        return TFolderListRequestContext::DoRequest(std::move(folderId), this);
    }

    TFuture<TListInstancesResult> TInstanceGroupClient::ListInstancesForFolder(TString folderId) {
        auto promise = NewPromise<TListInstancesResult>();
        ListInstanceGroups(std::move(folderId))
            .Subscribe([promise, self = IInstanceGroupClientPtr(this)] (auto f) mutable {
                try {
                    auto&& v = f.GetValue();
                    if (v.Success()) {
                        TMultipleRequestContext::DoRequest(v.Value(), self).Subscribe([=] (auto f) mutable {
                            promise.SetValue(f.ExtractValue());
                        });
                    } else {
                        promise.SetValue(TListInstancesResult::FromError(v.Error()));
                    }
                } catch (...) {
                    promise.SetValue(TListInstancesResult::FromError(CurrentExceptionMessage()));
                }
            });

        return promise.GetFuture();
    }
} // namespace

    IInstanceGroupClientPtr CreateInstanceGroupClient(
        const TGrpcConfig& config,
        ITokenProviderPtr tokenProvider,
        NMonitoring::TMetricRegistry& registry,
        TString clientId)
    {
        auto threadPool = CreateGrpcThreadPool(config);
        auto sc = CreateGrpcServiceConnection<::NCloud::InstanceGroupService>(
            config,
            true,
            registry,
            std::move(threadPool),
            std::move(clientId));

        return ::MakeIntrusive<TInstanceGroupClient>(std::move(sc), std::move(tokenProvider));
    }
} // namespace NSolomon::NCloud
