#include "cluster.h"

#include <solomon/libs/cpp/host_resolver/host_resolver.h>

#include <util/generic/hash.h>
#include <util/random/random.h>
#include <util/stream/input.h>
#include <util/stream/output.h>
#include <util/string/builder.h>
#include <util/string/split.h>
#include <util/system/hostname.h>

using namespace yandex::monitoring::config;

namespace NSolomon {
namespace {
    const auto LOCAL_FQDN = FQDNHostName();

    struct THostnameMapper: public INodeMapper {
        ui32 GetId(TStringBuf fqdn) const override {
            // get the numeric identifier of the host in a cluster.
            // typical hostname pattern is `env-service-dc-123.mon.yandex.net`
            auto shortName = fqdn.Before('.');
            auto nodeIdStr = shortName.RAfter('-');

            ui32 nodeId = 0;
            if (TryFromString(nodeIdStr, nodeId)) {
                return nodeId;
            }

            if (fqdn == LOCAL_FQDN) {
                // developer notebooks can have different hostnames,
                // so use some uniq ID for them but different from Max<ui32>()
                return Max<ui32>() - 1;
            }

            ythrow yexception() << "unable to parse node id from: '" << nodeIdStr << "'";
        }
    };

    auto DEFAULT_MAPPER = MakeHolder<THostnameMapper>();

    class TSingleHostCluster: public IClusterMap {
    public:
        TSingleHostCluster()
            : ThisNode_(FQDNHostName(), 0, 0)
        {
        }
            std::optional<TClusterNode> NodeByFqdn(const TString& fqdn) const override {
                return fqdn == ThisNode_.Fqdn ? std::optional{ThisNode_} : std::nullopt;
            }
            std::optional<TClusterNode> NodeById(i32 id) const override {
                return id == ThisNode_.NodeId ? std::optional{ThisNode_} : std::nullopt;
            }

            THashSet<TClusterNode> Nodes() const override {
                return {ThisNode_};
            }

            size_t Size() const override {
                return 1;
            }

            TClusterNode Any() const override {
                return ThisNode_;
            }

            TClusterNode Local() const override {
                return ThisNode_;
            }

    private:
        TClusterNode ThisNode_;
    };

    std::pair<TStringBuf, ui16> GetHostPort(TStringBuf hostPort) {
        TStringBuf host, portStr;
        ui16 port{0};
        Y_ENSURE(hostPort.TrySplit(':', host, portStr), "Cannot parse host and port from " << hostPort);
        Y_ENSURE(TryFromString(portStr, port), "Cannot parse port from " << portStr);
        return {host, port};
    }
} // namespace

    TClusterNode::TClusterNode(TString fqdn, i32 nodeId, ui16 port)
        : Fqdn{std::move(fqdn)}
        , Endpoint{TStringBuilder() << Fqdn << ':' << port}
        , Port{port}
        , NodeId{nodeId}
    {
        Y_ENSURE(!Fqdn.empty(), "fqdn cannot be empty");
    }

    bool TClusterNode::IsUnknown() const {
        return NodeId == UNKNOWN;
    }

    IOutputStream& TClusterNode::operator<<(IOutputStream& out) const {
        return out << "Fqdn: " << Fqdn << ", Endpoint: " << Endpoint << ", Port: " << Port << ", NodeId: " << NodeId;
    }

    bool IsLocal(const TClusterNode& loc) {
        return loc.Fqdn == LOCAL_FQDN;
    }

    std::optional<TClusterNode> TClusterMapBase::NodeByFqdn(const TString& fqdn) const {
        // XXX: do we need sublinear search here?
        // TODO: SOLOMON-5654
        auto it = FindIf(Nodes_, [&] (auto&& loc) {
            return loc.Fqdn == fqdn;
        });

        if (it == Nodes_.end()) {
            return {};
        }

        return *it;
    }

    std::optional<TClusterNode> TClusterMapBase::NodeById(i32 id) const {
        // TODO: SOLOMON-5654
        auto it = FindIf(Nodes_, [&] (auto&& loc) {
            return loc.NodeId == id;
        });

        if (it == Nodes_.end()) {
            return {};
        }

        return *it;
    }

    TClusterNode TClusterMapBase::Any() const {
        auto it = Nodes_.begin();
        for (size_t n = RandomNumber(Nodes_.size()); n; --n) {
            ++it;
        }
        Y_VERIFY(it != Nodes_.end());
        return *it;
    }

    TClusterNode TClusterMapBase::Local() const {
        return LocalNode_;
    }

    TIntrusivePtr<TClusterMapBase> TClusterMapBase::Load(TArrayRef<TStringBuf> addrs, INodeMapper* nodeMapper) {
        nodeMapper = nodeMapper
            ? nodeMapper
            : DEFAULT_MAPPER.Get();

        auto clusterMap = MakeIntrusive<TClusterMapBase>();

        auto addNode = [&] (TStringBuf fqdn, ui16 port) {
            auto [it, isNew] = clusterMap->Nodes_.emplace(TString{fqdn}, nodeMapper->GetId(fqdn), port);
            Y_VERIFY_DEBUG(isNew || it->Port != port, "Node %s with port %u is already present in the cluster", TString{fqdn}.c_str(), port);
            if (IsLocal(*it)) {
                clusterMap->LocalNode_ = *it;
            }
        };

        for (auto addr: addrs) {
            TStringBuf scheme, hostPort;
            if (!addr.TrySplit(TStringBuf("://"), scheme, hostPort)) {
                hostPort = addr;
            }

            if (scheme == TStringBuf("conductor_group")) {
                THashSet<TString> addrs;
                ConductorGroupResolver()->Resolve(addr, &addrs);

                for (auto&& addr: addrs) {
                    auto&& [fqdn, port] = GetHostPort(addr);
                    addNode(fqdn, port);
                }
            } else if (scheme.Empty()) {
                auto&& [host, port] = GetHostPort(hostPort);
                auto fqdn = host == TStringBuf("localhost")
                    ? FQDNHostName()
                    : ToString(host);

                    addNode(fqdn, port);
            } else {
                ythrow yexception() << "Unsupported scheme " << scheme;
            }
        }

        return clusterMap;
    }

    TIntrusivePtr<TClusterMapBase> TClusterMapBase::Load(TStringBuf data, INodeMapper* nodeMapper) {
        TVector<TStringBuf> addrs;
        StringSplitter(data)
            .SplitByFunc([](char ch) -> bool { return ch == ' ' || ch == '\t' || ch == '\n'; })
            .SkipEmpty()
            .Collect(&addrs);

        return Load(addrs, nodeMapper);
    }

    TIntrusivePtr<TClusterMapBase> TClusterMapBase::Load(IInputStream& is, INodeMapper* mapper) {
        TString data = is.ReadAll();
        return Load(data, mapper);
    }

    TIntrusivePtr<TClusterMapBase> TClusterMapBase::Load(const StaticClusterMapping& staticCluster) {
        auto clusterMap = MakeIntrusive<TClusterMapBase>();

        for (auto& node: staticCluster.nodes()) {
            auto fqdn = node.host() == TStringBuf("localhost")
                ? FQDNHostName()
                : ToString(node.host());
            auto nodeId = node.node_id();
            auto port = node.port();

            auto [it, isNew] = clusterMap->Nodes_.emplace(fqdn, nodeId, port);
            Y_VERIFY_DEBUG(
                isNew || it->Port != port,
                "Node %s with port %u is already present in the cluster",
                TString{fqdn}.c_str(), port
            );

            if (IsLocal(*it)) {
                clusterMap->LocalNode_ = *it;
            }
        }

        return clusterMap;
    }

    THashSet<TClusterNode> TClusterMapBase::Nodes() const {
        return Nodes_;
    }

    IClusterMapPtr LoadCluster(IInputStream& is) {
        return TClusterMapBase::Load(is);
    }

    IClusterMapPtr LoadCluster(const StaticClusterMapping& staticCluster) {
        return TClusterMapBase::Load(staticCluster);
    }

    IClusterMapPtr CreateSingleNodeCluster() {
        return new TSingleHostCluster{};
    }

    THolder<INodeMapper> CreateDefaultMapper() {
        return MakeHolder<THostnameMapper>();
    }

    yandex::solomon::config::rpc::TGrpcClientConfig ConstructGrpcClientConfigFromCluster(const IClusterMap& cluster) {
        yandex::solomon::config::rpc::TGrpcClientConfig grpcClientConfig;

        for (auto& node: cluster.Nodes()) {
            if (node.NodeId != cluster.Local().NodeId) {
                grpcClientConfig.add_addresses(node.Endpoint);
            }
        }

        return grpcClientConfig;
    }
} // namespace NSolomon::NFetcher

template <>
void Out<NSolomon::TClusterNode>(IOutputStream& os, const NSolomon::TClusterNode& node) {
    os << '[' << node.NodeId << TStringBuf("] ") << node.Fqdn << ':' << node.Port;
}

template <>
void Out<NSolomon::IClusterMap>(IOutputStream& os, const NSolomon::IClusterMap& cluster) {
    for (const auto& node: cluster.Nodes()) {
        os << node << '\n';
    }
}
