#pragma once

#include <solomon/protos/configs/rpc/rpc_config.pb.h>
#include <solomon/protos/configs/static_cluster.pb.h>

#include <util/generic/array_ref.h>
#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/generic/string.h>
#include <util/stream/fwd.h>
#include <util/string/builder.h>
#include <util/system/types.h>

#include <optional>

namespace NSolomon {
    struct IHostResolver;
    struct TClusterNode {
        static constexpr i32 UNKNOWN{-1};

        TClusterNode() = default;
        TClusterNode(TString fqdn, i32 nodeId, ui16 port);

        TString Fqdn;
        /**
         * Fqdn:Port
         */
        TString Endpoint;
        ui16 Port{0};

        // this way we can avoid comparing strings and compare two integers instead
        i32 NodeId{UNKNOWN};

        bool IsUnknown() const;
        IOutputStream& operator<<(IOutputStream& out) const;
    };
} // namespace NSolomon::NFetcher

template <>
struct THash<NSolomon::TClusterNode> {
    size_t operator()(const NSolomon::TClusterNode& loc) const {
        return ComputeHash(loc.Endpoint);
    }
};

namespace NSolomon {
    extern const TClusterNode UNKNOWN_NODE;

    inline bool operator==(const TClusterNode& lhs, const TClusterNode& rhs) {
        if (lhs.IsUnknown() || rhs.IsUnknown()) {
            return false;
        }

        return lhs.Endpoint == rhs.Endpoint && lhs.NodeId == rhs.NodeId;
    }

    inline bool operator!=(const TClusterNode& lhs, const TClusterNode& rhs) {
        return !(rhs == lhs);
    }

    bool IsLocal(const TClusterNode& loc);

    class IClusterMap: public TThrRefBase {
    public:
        virtual std::optional<TClusterNode> NodeByFqdn(const TString& fqdn) const = 0;
        virtual std::optional<TClusterNode> NodeById(i32 id) const = 0;
        virtual THashSet<TClusterNode> Nodes() const = 0;
        virtual size_t Size() const = 0;

        virtual TClusterNode Any() const = 0;
        virtual TClusterNode Local() const = 0;
    };

    // allows to customize the way hostnames are mapped into numeric IDs
    struct INodeMapper {
        virtual ~INodeMapper() = default;
        virtual ui32 GetId(TStringBuf hostname) const = 0;
    };

    class TClusterMapBase: public IClusterMap {
    public:
        std::optional<TClusterNode> NodeByFqdn(const TString& fqdn) const override;
        std::optional<TClusterNode> NodeById(i32 id) const override;
        THashSet<TClusterNode> Nodes() const override;
        size_t Size() const override {
            return Nodes_.size();
        }

        TClusterNode Any() const override;
        TClusterNode Local() const override;

        // if mapper is not set, then by default we'll extract node ID from hostname.
        // E.g. solomon-pre-fetcher-man-000 will be assigned node ID 0
        static TIntrusivePtr<TClusterMapBase> Load(IInputStream&, INodeMapper* nodeMapper = nullptr);
        static TIntrusivePtr<TClusterMapBase> Load(TStringBuf, INodeMapper* nodeMapper = nullptr);
        static TIntrusivePtr<TClusterMapBase> Load(TArrayRef<TStringBuf> addresses, INodeMapper* nodeMapper = nullptr);
        static TIntrusivePtr<TClusterMapBase> Load(const yandex::monitoring::config::StaticClusterMapping& staticCluster);

    protected:
        THashSet<TClusterNode> Nodes_;
        TClusterNode LocalNode_;
    };

    using IClusterMapPtr = TIntrusivePtr<IClusterMap>;

    IClusterMapPtr LoadCluster(IInputStream& is);
    IClusterMapPtr LoadCluster(const yandex::monitoring::config::StaticClusterMapping& staticCluster);

    // Creates the cluster map with the only node being current host and node ID 0
    IClusterMapPtr CreateSingleNodeCluster();

    // picks node id as a number from hostname of form "something-000.*"
    THolder<INodeMapper> CreateDefaultMapper();

    template <typename TConfig>
    inline IClusterMapPtr MakeClusterMapFromConfig(const TConfig& config) {
        if (auto proto = config.dynamic_cluster_mapping(); !proto.empty()) {
            TStringBuilder sb;
            for (auto&& s: proto) {
                sb << s << '\n';
            }

            return TClusterMapBase::Load(sb);
        } else if (config.has_static_cluster_mapping()) {
            return LoadCluster(config.static_cluster_mapping());
        } else {
            return CreateSingleNodeCluster();
        }
    }

    yandex::solomon::config::rpc::TGrpcClientConfig ConstructGrpcClientConfigFromCluster(const IClusterMap& cluster);
} // namespace NSolomon
