#include <solomon/libs/cpp/cluster_map/cluster.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/stream/mem.h>
#include <util/string/cast.h>
#include <util/system/hostname.h>

#include <iostream>

using namespace testing;
using namespace NSolomon;

namespace {

struct IncMapper: INodeMapper {
    ui32 GetId(TStringBuf) const override {
        return Idx++;
    }

    mutable ui32 Idx{0};
};

} // namespace

MATCHER_P2(NodeLike, pattern, port, std::string(negation ? "isn't like " : "is like ") + pattern + ':' + ToString(port)) {
    return arg.Port == port && Matches(MatchesRegex(pattern))(arg.Fqdn);
}

TEST(TClusterMapTest, ClusterMapLoadOk) {
    const auto hostname = FQDNHostName();

    {
        const TString data = TStringBuilder() << hostname << ":80";
        auto clusterMap = TClusterMapBase::Load(data);

        ASSERT_THAT(clusterMap->Nodes(), SizeIs(1));
        ASSERT_THAT(clusterMap->Local().Fqdn, StrEq(hostname));
    }

    {
        IncMapper mapper;

        const TString data = TStringBuilder() << R"(
            solomon-pre-fetcher-man-000:80
            solomon-pre-fetcher-man-001.search.yandex.net:80
            solomon-pre-fetcher-man-002.search.yandex.net:80
        )"  << hostname << ":80" << Endl;

        auto clusterMap = TClusterMapBase::Load(data, &mapper);
        ASSERT_THAT(clusterMap->Nodes(), UnorderedElementsAre(
            TClusterNode{"solomon-pre-fetcher-man-000", 0, 80},
            TClusterNode{"solomon-pre-fetcher-man-001.search.yandex.net", 1, 80},
            TClusterNode{"solomon-pre-fetcher-man-002.search.yandex.net", 2, 80},
            TClusterNode{hostname, 3, 80}
        ));
    }

    // without id 0, with holes, and with the default mapper
    {
        const TString data = TStringBuilder() << R"(
            solomon-pre-fetcher-man-001.search.yandex.net:80
            solomon-pre-fetcher-man-003.search.yandex.net:80
        )";

        auto clusterMap = TClusterMapBase::Load(data);
        ASSERT_THAT(clusterMap->Nodes(), UnorderedElementsAre(
            TClusterNode{"solomon-pre-fetcher-man-001.search.yandex.net", 1, 80},
            TClusterNode{"solomon-pre-fetcher-man-003.search.yandex.net", 3, 80}
        ));
    }
}

TEST(TClusterMapTest, FromConductorGroup) {
    const TString data = TStringBuilder() << "conductor_group://solomon_test_sts:1111";
    auto clusterMap = TClusterMapBase::Load(data);
    auto nodes = clusterMap->Nodes();
    ASSERT_THAT(nodes, Each(NodeLike(R"(solomon-test-sts-vla-.+\.search.yandex.net)", 1111)));
}

TEST(TClusterMapTest, ClusterMapLoadBrokenFormat) {
    constexpr auto noPort = R"(solomon-pre-fetcher-man-000)";
    constexpr auto unknownScheme = R"(foobar://solomon-pre-fetcher-man-000:80)";

    auto load = [] (auto&& data) {
        TMemoryInput in{data};
        auto clusterMap = LoadCluster(in);
        return clusterMap->Nodes();
    };

    ASSERT_THROW(load(noPort), yexception);
    ASSERT_THROW(load(unknownScheme), yexception);
}

TEST(TNodeMapperTest, DefaultMapper) {
    auto mapper = CreateDefaultMapper();
    auto id = mapper->GetId("solomon-test-sts-vla-001.search.yandex.net");
    ASSERT_THAT(id, Eq(1u));
}

TEST(TGrpcClientConfigFromCluster, HappyPath) {
    TStringBuf localhost = "localhost";
    IncMapper mapper;

    const TString data = TStringBuilder() << R"(
        solomon-pre-fetcher-man-000:80
        solomon-pre-fetcher-man-001.search.yandex.net:81
        solomon-pre-fetcher-man-002.search.yandex.net:82
    )"  << localhost << ":83" << Endl;

    auto clusterMapPtr = TClusterMapBase::Load(data, &mapper);
    auto grpcClient = ConstructGrpcClientConfigFromCluster(*clusterMapPtr);

    ASSERT_THAT(grpcClient.addresses(), UnorderedElementsAre(
        TString{"solomon-pre-fetcher-man-000:80"},
        TString{"solomon-pre-fetcher-man-001.search.yandex.net:81"},
        TString{"solomon-pre-fetcher-man-002.search.yandex.net:82"}
    ));
}
