#pragma once

#include "node_states.h"

#include <solomon/libs/cpp/cluster_membership/api/cluster_membership_grpc_service.grpc.pb.h>

#include <util/generic/vector.h>
#include <util/generic/yexception.h>

namespace NSolomon::NClusterMembership {

struct TMembershipUpdate {
    TMembershipUpdate(TString supervisor, ui32 incarnationNumber, ENodeState state)
        : Supervisor{std::move(supervisor)}
        , IncarnationNumber{incarnationNumber}
        , State{state}
    {}

    TString Supervisor;
    ui32 IncarnationNumber;
    ENodeState State;
};

struct TGossip: public TAtomicRefCount<TGossip> {
    TGossip(TString target, TString supervisor, ui32 incarnationNumber, ENodeState state)
        : Target{std::move(target)}
        , Update{std::move(supervisor), incarnationNumber, state}
    {}

    TString Target;
    TMembershipUpdate Update;
    ui32 PiggybackCnt{0};
};

using TGossipPtr = TIntrusivePtr<TGossip>;
using TGossips = TVector<TGossipPtr>;

inline yandex::monitoring::cluster_membership::NodeState ConvertNodeStateToProto(ENodeState state) {
    switch (state) {
        case ENodeState::Unknown:
            return yandex::monitoring::cluster_membership::NODESTATE_UNSPECIFIED;
        case ENodeState::Alive:
            return yandex::monitoring::cluster_membership::NODESTATE_ALIVE;
        case ENodeState::Suspicious:
            return yandex::monitoring::cluster_membership::NODESTATE_SUSPICIOUS;
        case ENodeState::Dead:
            return yandex::monitoring::cluster_membership::NODESTATE_DEAD;
    }
}

inline ENodeState ConvertNodeStateFromProto(yandex::monitoring::cluster_membership::NodeState state) {
    switch (state) {
        case yandex::monitoring::cluster_membership::NODESTATE_UNSPECIFIED:
            return ENodeState::Unknown;
        case yandex::monitoring::cluster_membership::NODESTATE_ALIVE:
            return ENodeState::Alive;
        case yandex::monitoring::cluster_membership::NODESTATE_SUSPICIOUS:
            return ENodeState::Suspicious;
        case yandex::monitoring::cluster_membership::NODESTATE_DEAD:
            return ENodeState::Dead;
        default:
            ythrow yexception() << "unsupported state value: " << yandex::monitoring::cluster_membership::NodeState_Name(state);
    }
}

template <class TMessage>
inline void InflateMessageWithDissemination(TMessage* message, const TAtomicSharedPtr<TGossips>& gossips) {
    if (gossips && !gossips->empty()) {
        for (auto& gossipPtr: *gossips) {
            auto* g = message->mutable_dissemination()->add_gossips();

            g->set_supervisor(gossipPtr->Update.Supervisor);
            g->set_target(gossipPtr->Target);
            g->mutable_update()->set_state(ConvertNodeStateToProto(gossipPtr->Update.State));
            g->mutable_update()->set_incarnation_number(gossipPtr->Update.IncarnationNumber);
        }
    }
}

} // namespace NSolomon::NClusterMembership
