#include "grpc_client.h"

#include <solomon/libs/cpp/cluster_membership/api/cluster_membership_grpc_service.grpc.pb.h>
#include <solomon/libs/cpp/cluster_membership/config/cluster_membership_config.pb.h>

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/string/builder.h>
#include <util/string/cast.h>

namespace NSolomon::NClusterMembership {
namespace {

using namespace NMonitoring;
using namespace NThreading;
using namespace grpc;
using namespace yandex::monitoring::cluster_membership;

using NGrpc::TGRpcClientConfig;
using NGrpc::TGRpcClientLow;
using NGrpc::TGrpcStatus;
using NGrpc::TServiceConnection;
using yandex::solomon::config::rpc::TGrpcClientConfig;

// Gossips
TAckResponse ConvertAckResponseFromProto(AckResponse&& resp) {
    TAckResponse native;
    native.Address = resp.address();
    native.IncarnationNumber = resp.incarnation_number();
    native.Success = resp.success();

    if (resp.has_dissemination() && resp.dissemination().gossipsSize() > 0) {
        native.Gossips = new TGossips{};

        for (auto& gossipProto: resp.dissemination().gossips()) {
            native.Gossips->emplace_back(
                new TGossip(
                    gossipProto.target(),
                    gossipProto.supervisor(),
                    gossipProto.update().incarnation_number(),
                    ConvertNodeStateFromProto(gossipProto.update().state())));
        }
    }

    return native;
}

class TGrpcSingleHostClient: public IClusterMembershipSingleHostClient {
public:
    TGrpcSingleHostClient(
            IMetricRegistry&,
            std::unique_ptr<TGrpcServiceConnection<ClusterMembershipService>> connection)
        : Connection_{std::move(connection)}
    {
    }

    TAsyncAckResponse Ping(
            TString address,
            TAtomicSharedPtr<TGossips> gossips) noexcept override
    {
        PingRequest req;
        req.set_address(address);
        InflateMessageWithDissemination(&req, gossips);

        auto promise = NewPromise<TAsyncAckResponse::value_type>();

        auto cb = [promise] (TGrpcStatus&& status, AckResponse&& result) mutable {
            if (!status.Ok()) {
                promise.SetValue(std::make_unique<TAckResponseOrError>(status));
            } else {
                promise.SetValue(std::make_unique<TAckResponseOrError>(ConvertAckResponseFromProto(std::move(result))));
            }
        };

        Connection_->Request<PingRequest, AckResponse>(
            std::move(req),
            std::move(cb),
            &ClusterMembershipService::Stub::AsyncPing);

        return promise;
    };

    void Stop(bool) noexcept override {
    }

private:
    std::unique_ptr<TGrpcServiceConnection<ClusterMembershipService>> Connection_;
};

} // namespace

IClusterMembershipClientPtr CreateClusterMembershipGrpcClient(const TGrpcClientConfig& conf, TMetricRegistry& registry, TString clientId) {
    return std::make_shared<TGrpcClusterClientBase<ClusterMembershipService, IClusterMembershipSingleHostClient>>(
        conf,
        false,
        registry,
        [](IMetricRegistry& registry, std::unique_ptr<TGrpcServiceConnection<ClusterMembershipService>> connection) {
            return std::make_unique<TGrpcSingleHostClient>(registry, std::move(connection));
        },
        std::move(clientId)
    );
}

} // namespace NSolomon::NClusterMembership
