#include "common.h"
#include "events.h"
#include "grpc_service.h"

#include <solomon/libs/cpp/cluster_membership/api/cluster_membership_grpc_service.grpc.pb.h>

using namespace NActors;
using namespace NMonitoring;
using namespace yandex::monitoring::cluster_membership;

namespace NSolomon::NClusterMembership {
namespace {

SOLOMON_RPC_HANDLER(yandex::monitoring::cluster_membership::ClusterMembershipService, Ping) {
public:
    explicit TPingHandler(NActors::TActorId clusterMembershipActor)
        : ClusterMembershipActor_{clusterMembershipActor}
    {}

    void Handle(const NActors::TActorContext&) override {
        auto* req = GetRequest();

        TAtomicSharedPtr<TGossips> gossips;
        if (req->has_dissemination() && req->dissemination().gossipsSize() > 0) {
            gossips = new TGossips{};

            for (auto& gossipProto: req->dissemination().gossips()) {
                gossips->emplace_back(
                        new TGossip(
                                gossipProto.target(),
                                gossipProto.supervisor(),
                                gossipProto.update().incarnation_number(),
                                ConvertNodeStateFromProto(gossipProto.update().state())));
            }
        }

        // TODO: validate and check a req
        Send(ClusterMembershipActor_, new TGossipEvents::TPing{req->address(), std::move(gossips)});
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TGossipEvents::TAck, OnAck);
        }
    }

private:
    void OnAck(TGossipEvents::TAck::TPtr& ev) {
        auto& ackOrError = ev->Get()->Ack;

        if (ackOrError.Fail()) {
            GetRequestCtx()->ReplyError(grpc::StatusCode::INTERNAL, ackOrError.Error().MessageString());
        } else {
            using AckResponse = yandex::monitoring::cluster_membership::AckResponse;
            auto* arena = GetRequestCtx()->GetArena();
            auto* reply = google::protobuf::Arena::CreateMessage<AckResponse>(arena);

            auto& ack = ackOrError.Value();
            reply->set_address(ack.Address);
            reply->set_incarnation_number(ack.IncarnationNumber);
            reply->set_success(ack.Success);
            InflateMessageWithDissemination(reply, ack.Gossips);

            GetRequestCtx()->Reply(reply);
        }

        PassAway();
    }

    void OnTimeout() {
        GetRequestCtx()->ReplyError(grpc::StatusCode::DEADLINE_EXCEEDED, "request timeout");
        PassAway();
    }

private:
    NActors::TActorId ClusterMembershipActor_;
};

class TClusterMembershipService final: public NSolomon::NGrpc::TService<ClusterMembershipService> {
public:
    TClusterMembershipService(
            TActorSystem& actorSystem,
            TMetricRegistry& registry,
            ui32 executorPool,
            TActorId clusterMembershipActorId)
        : TService(actorSystem, registry, executorPool)
        , ClusterMembershipActorId_(clusterMembershipActorId)
    {
    }

private:
    void InitHandlers(grpc::ServerCompletionQueue* cq) override {
        RegisterHandler<TPingHandler>(cq, ClusterMembershipActorId_);
    }

private:
    TActorId ClusterMembershipActorId_;
};

} // namespace

TIntrusivePtr<::NGrpc::IGRpcService> CreateClusterMembershipService(
        TActorSystem& actorSystem,
        TMetricRegistry& registry,
        ui32 executorPool,
        TActorId clusterMembershipActorId)
{
    return MakeIntrusive<TClusterMembershipService>(actorSystem, registry, executorPool, clusterMembershipActorId);
}

} // namespace NSolomon::NClusterMembership
