#pragma once

#include <solomon/libs/cpp/ydb/config/ydb_config.pb.h>

#include <library/cpp/monlib/metrics/fwd.h>
#include <library/cpp/threading/future/future.h>

#include <util/generic/ptr.h>

#include <optional>

namespace NYdb {
class TDriver;
} // namespace NYdb

namespace NSolomon::NDb {
namespace NModel {

class TAgentConfig;
class TClusterConfig;
class TProjectConfig;
class TProviderConfig;
class TServiceConfig;
class TShardConfig;

} // namespace NModel

struct TShardTables {
    TString ShardTablePath;
    TString NumPcsTablePath;
    TString NumIdPcsTablePath;
};

using TAsyncProjectConfig = NThreading::TFuture<std::optional<NModel::TProjectConfig>>;
using TAsyncProviderConfig = NThreading::TFuture<std::optional<NModel::TProviderConfig>>;
using TAsyncServiceConfig = NThreading::TFuture<std::optional<NModel::TServiceConfig>>;
using TAsyncClusterConfig = NThreading::TFuture<std::optional<NModel::TClusterConfig>>;
using TAsyncShardConfig = NThreading::TFuture<std::optional<NModel::TShardConfig>>;
using TAsyncAgentConfigs = NThreading::TFuture<TVector<NModel::TAgentConfig>>;
using TAsyncClusterConfigs = NThreading::TFuture<TVector<NModel::TClusterConfig>>;
using TAsyncProjectConfigs = NThreading::TFuture<TVector<NModel::TProjectConfig>>;
using TAsyncProviderConfigs = NThreading::TFuture<TVector<NModel::TProviderConfig>>;
using TAsyncServiceConfigs = NThreading::TFuture<TVector<NModel::TServiceConfig>>;
using TAsyncShardConfigs = NThreading::TFuture<TVector<NModel::TShardConfig>>;
using TAsyncVoid = NThreading::TFuture<void>;
using TAsyncBool = NThreading::TFuture<bool>;

class IClusterConfigDao: public TThrRefBase {
public:
    virtual TAsyncVoid CreateTable() = 0;
    virtual TAsyncVoid DropTable() = 0;

    virtual TAsyncVoid Insert(const NModel::TClusterConfig& model) = 0;
    virtual TAsyncVoid Delete(const TString& clusterId, const TString& projectId) = 0;

    virtual TAsyncClusterConfig GetByProject(const TString& clusterId, const TString& projectId) = 0;
    virtual TAsyncClusterConfig GetByName(const TString& clusterName, const TString& projectId) = 0;
    virtual TAsyncClusterConfigs GetAll() = 0;
};

class IShardConfigDao: public TThrRefBase {
public:
    virtual TAsyncVoid CreateTable() = 0;
    virtual TAsyncVoid DropTable() = 0;

    virtual TAsyncVoid Insert(const NModel::TShardConfig& model) = 0;
    virtual TAsyncVoid Delete(const TString& shardId, const TString& projectId) = 0;

    virtual TAsyncShardConfigs GetAll() = 0;
    virtual TAsyncShardConfig GetById(const TString& shardId, const TString& projectId) = 0;
};

class IServiceConfigDao: public TThrRefBase {
public:
    virtual TAsyncVoid CreateTable() = 0;
    virtual TAsyncVoid DropTable() = 0;

    virtual TAsyncVoid Insert(const NModel::TServiceConfig& model) = 0;
    virtual TAsyncVoid Delete(const TString& serviceId, const TString& projectId) = 0;

    virtual TAsyncServiceConfig GetByProject(const TString& serviceId, const TString& projectId) = 0;
    virtual TAsyncServiceConfig GetByName(const TString& serviceName, const TString& projectId) = 0;
    virtual TAsyncServiceConfigs GetAll() = 0;
};

class IProjectConfigDao: public TThrRefBase {
public:
    virtual TAsyncVoid CreateTable() = 0;
    virtual TAsyncVoid DropTable() = 0;

    virtual TAsyncBool Exists(const TString& projectId) = 0;
    virtual TAsyncVoid Insert(const NModel::TProjectConfig& model) = 0;
    virtual TAsyncVoid Delete(const TString& projectId) = 0;
    virtual TAsyncProjectConfig GetById(const TString& projectId) = 0;
    virtual TAsyncProjectConfigs GetAll() = 0;
};

class IAgentConfigDao: public TThrRefBase {
public:
    virtual TAsyncVoid CreateTable() = 0;
    virtual TAsyncVoid DropTable() = 0;

    virtual TAsyncVoid InsertOrUpdate(const NModel::TAgentConfig& model) = 0;
    virtual TAsyncVoid Delete(const TString& url, const TString& provider) = 0;
    virtual TAsyncVoid DeleteByProvider(const TString& provider) = 0;
    virtual TAsyncVoid DeleteObsolete(TInstant deadline) = 0;

    virtual TAsyncAgentConfigs GetAll() = 0;
};

class IProviderConfigDao: public TThrRefBase {
public:
    virtual TAsyncVoid CreateTable() = 0;
    virtual TAsyncVoid DropTable() = 0;

    virtual TAsyncBool Exists(const TString& providerId) = 0;
    virtual TAsyncVoid Insert(const NModel::TProviderConfig& model) = 0;
    virtual TAsyncVoid Delete(const TString& providerId) = 0;
    virtual TAsyncProviderConfig GetById(const TString& providerId) = 0;
    virtual TAsyncProviderConfigs GetAll() = 0;
};

using IClusterConfigDaoPtr = TIntrusivePtr<IClusterConfigDao>;
using IShardConfigDaoPtr = TIntrusivePtr<IShardConfigDao>;
using IServiceConfigDaoPtr = TIntrusivePtr<IServiceConfigDao>;
using IAgentConfigDaoPtr = TIntrusivePtr<IAgentConfigDao>;
using IProjectConfigDaoPtr = TIntrusivePtr<IProjectConfigDao>;
using IProviderConfigDaoPtr = TIntrusivePtr<IProviderConfigDao>;

class IDbConnection: public TThrRefBase {
public:
    virtual IClusterConfigDaoPtr CreateClusterDao(const TString& path) = 0;
    virtual IShardConfigDaoPtr CreateShardDao(const TShardTables& path) = 0;
    virtual IServiceConfigDaoPtr CreateServiceDao(const TString& path) = 0;
    virtual IAgentConfigDaoPtr CreateAgentDao(const TString& path) = 0;
    virtual IProjectConfigDaoPtr CreateProjectDao(const TString& path) = 0;
    virtual IProviderConfigDaoPtr CreateProviderDao(const TString& path) = 0;
};

using IDbConnectionPtr = TIntrusivePtr<IDbConnection>;

IDbConnectionPtr CreateYdbConnection(NYdb::TDriver driver, NMonitoring::TMetricRegistry& registry);
IDbConnectionPtr CreateYdbConnection(const NYdb::TDriver& driver, const TSessionPoolConfig& poolConfig, NMonitoring::TMetricRegistry& registry);
IDbConnectionPtr CreateYdbConnection(const TYdbConfig& config, NMonitoring::TMetricRegistry& registry);

} // namespace NSolomon::NDb
