#include "aggregation_settings.h"
#include "json.h"

#include <ostream>

namespace NSolomon::NDb::NModel {
namespace {

template <typename TStream>
void Write(TStream& os, const TAggregationRule& rule) {
    os << "Conditions: [";
    for (size_t i = 0, size = rule.Conditions.size(); i < size; ++i) {
        if (i > 0) {
            os << ", ";
        }
        os << '\'' << rule.Conditions[i] << '\'';
    }

    os << "], Targets: [";
    for (size_t i = 0, size = rule.Targets.size(); i < size; ++i) {
        if (i > 0) {
            os << ", ";
        }
        os << '\'' << rule.Targets[i] << '\'';
    }

    os << "], Function: " << ToString(rule.Function);
}

template <typename TStream>
void Write(TStream& os, const TAggregationSettings& settings) {
    os << "Rules: [";
    for (size_t i = 0, size = settings.Rules.size(); i < size; ++i) {
        if (i > 0) {
            os << ", ";
        }
        os << '{' << settings.Rules[i] << '}';
    }
    os << "], Enabled: " << settings.Enabled;
    os << ", MemOnly: " << settings.MemOnly;
}

} // namespace

std::optional<TAggregationRule> TAggregationRule::FromJsonStr(TStringBuf data) {
    return NModel::FromJsonStr<TAggregationRule>(data);
}

TAggregationRule TAggregationRule::FromJsonVal(const NJson::TJsonValue& json) {
    TAggregationRule rule;

    EAggregationFunction function;
    if (const auto& str = json["aggr"].GetString(); !str.empty() && TryFromString(str, function)) {
        rule.Function = function;
    }

    for (const auto& condJson: json["cond"].GetArray()) {
        rule.Conditions.emplace_back(condJson.GetString());
    }

    for (const auto& targetJson: json["target"].GetArray()) {
        rule.Targets.emplace_back(targetJson.GetString());
    }

    return rule;
}

TString TAggregationRule::ToJsonStr() const {
    return NModel::ToJsonStr<TAggregationRule>(*this);
}

void TAggregationRule::ToJsonVal(NJsonWriter::TBuf& json) const {
    json.BeginObject();
    json.WriteKey("aggr").WriteString(ToString(Function));
    {
        json.WriteKey("cond");
        json.BeginList();
        for (const auto& cond: Conditions) {
            json.WriteString(cond);
        }
        json.EndList();
    }
    {
        json.WriteKey("target");
        json.BeginList();
        for (const auto& target: Targets) {
            json.WriteString(target);
        }
        json.EndList();
    }
    json.EndObject();
}

IOutputStream& operator<<(IOutputStream& os, const TAggregationRule& rule) {
    Write(os, rule);
    return os;
}

std::ostream& operator<<(std::ostream& os, const TAggregationRule& rule) {
    Write(os, rule);
    return os;
}

std::optional<TAggregationSettings> TAggregationSettings::FromJsonStr(TStringBuf data) {
    return NModel::FromJsonStr<TAggregationSettings>(data);
}

TAggregationSettings TAggregationSettings::FromJsonVal(const NJson::TJsonValue& json) {
    TAggregationSettings settings;

    for (const auto& ruleJson: json["aggrRules"].GetArray()) {
        if (ruleJson.IsMap()) {
            settings.Rules.push_back(TAggregationRule::FromJsonVal(ruleJson));
        }
    }

    settings.Enabled = json["enabled"].GetBoolean();
    settings.MemOnly = json["memOnly"].GetBoolean();

    return settings;
}

TString TAggregationSettings::ToJsonStr() const {
    return NModel::ToJsonStr<TAggregationSettings>(*this);
}

void TAggregationSettings::ToJsonVal(NJsonWriter::TBuf& json) const {
    json.BeginObject();
    {
        json.WriteKey("aggrRules");
        json.BeginList();
        for (const auto& rule: Rules) {
            rule.ToJsonVal(json);
        }
        json.EndList();
    }

    json.WriteKey("enabled").WriteBool(Enabled);
    json.WriteKey("memOnly").WriteBool(MemOnly);
    json.EndObject();
}

IOutputStream& operator<<(IOutputStream& os, const TAggregationSettings& settings) {
    Write(os, settings);
    return os;
}

std::ostream& operator<<(std::ostream& os, const TAggregationSettings& settings) {
    Write(os, settings);
    return os;
}

} // namespace NSolomon::NDb::NModel
