#include "provider_config.h"

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/writer/json.h>

namespace NSolomon::NDb::NModel {

TProviderShardSettings TProviderShardSettings::FromString(const TString& data) {
    NJson::TJsonReaderConfig readerConfig;
    readerConfig.DontValidateUtf8 = true;

    NJson::TJsonValue jsonValue;

    try {
        NJson::ReadJsonTree(data, &readerConfig, &jsonValue, true);
    } catch (NJson::TJsonException ex) {
        // TODO(SOLOMON-5558):
        Cerr << ex.AsStrBuf();
        return TProviderShardSettings{};
    }

    TProviderShardSettings setts;
    setts.MetricConf = TMetricConf::FromJsonValue(jsonValue["metricConf"]);
    setts.MetricsTtlDays = jsonValue["metricTtlDays"].GetUInteger();
    setts.Grid = TDuration::Seconds(jsonValue["grid"].GetUInteger());
    setts.Interval = TDuration::Seconds(jsonValue["interval"].GetUInteger());

    return setts;
}

TString TProviderShardSettings::ToString() const {
    NJsonWriter::TBuf jsonOutput;
    jsonOutput.BeginObject();

    jsonOutput.WriteKey("metricConf")
            .BeginObject();
    MetricConf.AddToJsonOutput(jsonOutput);
    jsonOutput.EndObject();

    jsonOutput.WriteKey("metricTtlDays").WriteULongLong(MetricsTtlDays)
            .WriteKey("grid").WriteULongLong(Grid.Seconds())
            .WriteKey("interval").WriteULongLong(Interval.Seconds())
            ;

    jsonOutput.EndObject();
    return jsonOutput.Str();
}

TReferenceConf TReferenceConf::FromJson(const NJson::TJsonValue& jsonValue) {
    TReferenceConf result;
    result.Label = jsonValue["label"].GetString();
    {
        auto&& services = jsonValue["services"].GetArray();
        for (auto&& service: services) {
            result.Services.push_back(service.GetString());
        }
    }
    {
        auto&& types = jsonValue["types"].GetArray();
        for (auto&& type: types) {
            result.Types.push_back(type.GetString());
        }
    }
    result.CrossFolder = jsonValue["crossFolder"].GetBoolean();
    return result;
}

TVector<TReferenceConf> TReferenceConf::FromStringArray(const TString& data) {
    NJson::TJsonReaderConfig readerConfig;
    readerConfig.DontValidateUtf8 = true;

    NJson::TJsonValue jsonValue;
    NJson::ReadJsonTree(data, &readerConfig, &jsonValue, true);

    auto&& references = jsonValue.GetArray();
    TVector<TReferenceConf> result;
    for (auto&& reference: references) {
        result.push_back(TReferenceConf::FromJson(reference));
    }
    return result;
}

void TReferenceConf::ToJson(NJsonWriter::TBuf& json) const {
    json.BeginObject();

    json.WriteKey("label").WriteString(Label);
    {
        json.WriteKey("services");
        json.BeginList();
        for (const auto& service: Services) {
            json.WriteString(service);
        }
        json.EndList();
    }
    {
        json.WriteKey("types");
        json.BeginList();
        for (const auto& type: Types) {
            json.WriteString(type);
        }
        json.EndList();
    }
    json.WriteKey("crossFolder").WriteBool(CrossFolder);
    json.EndObject();
}

TString TReferenceConf::ToString() const {
    NJsonWriter::TBuf json;
    ToJson(json);

    return json.Str();
}

TString TReferenceConf::ToStringArray(const TVector<TReferenceConf>& data) {
    NJsonWriter::TBuf json;
    json.BeginList();

    for (auto& reference: data) {
        reference.ToJson(json);
    }

    json.EndList();
    return json.Str();
}

bool operator==(const TProviderShardSettings& lhs, const TProviderShardSettings& rhs) {
    return lhs.MetricConf == rhs.MetricConf
            && lhs.MetricsTtlDays == rhs.MetricsTtlDays
            && lhs.Grid == rhs.Grid
            && lhs.Interval == rhs.Interval
            ;
}

bool operator==(const TReferenceConf& lhs, const TReferenceConf& rhs) {
    return lhs.Label == rhs.Label
        && lhs.Services == rhs.Services
        && lhs.Types == rhs.Types
        && lhs.CrossFolder == rhs.CrossFolder
        ;
}

bool operator==(const TProviderConfig& lhs, const TProviderConfig& rhs) {
    return lhs.Id == rhs.Id
            && lhs.Description == rhs.Description
            && lhs.ShardSettings == rhs.ShardSettings
            && lhs.References == rhs.References
            && lhs.AbcService == rhs.AbcService
            && lhs.CloudId == rhs.CloudId
            && lhs.TvmDestId == rhs.TvmDestId
            && lhs.IamServiceAccountIds == rhs.IamServiceAccountIds
            && lhs.TvmServiceIds == rhs.TvmServiceIds
            && lhs.CreatedAt == rhs.CreatedAt
            && lhs.UpdatedAt == rhs.UpdatedAt
            && lhs.CreatedBy == rhs.CreatedBy
            && lhs.UpdatedBy == rhs.UpdatedBy
            && lhs.Version == rhs.Version
            ;
}

IOutputStream& operator<<(IOutputStream& os, const TReferenceConf& conf) {
    return os << conf.ToString();
}

IOutputStream& operator<<(IOutputStream& os, const TProviderConfig& conf) {
    os << "Id: " << conf.Id << '\n'
       << "Description: " << conf.Description << '\n'
       << "ShardSettings: " << conf.ShardSettings.ToString() << '\n'
       << "ReferenceSettings: " << NModel::TReferenceConf::ToStringArray(conf.References) << '\n'
       << "AbcService: " << conf.AbcService << '\n'
       << "CloudId: " << conf.CloudId << '\n'
       << "TvmDestId: " << conf.TvmDestId << '\n';

    os << "IamServiceAccountIds: [";
    for (size_t i = 0; i != conf.IamServiceAccountIds.size(); ++i) {
        if (i > 0) {
            os << ", ";
        }

        os << '"' << conf.IamServiceAccountIds[i] << '"';
    }
    os << ']';

    os << "TvmServiceIds: [";
    for (size_t i = 0; i != conf.TvmServiceIds.size(); ++i) {
        if (i > 0) {
            os << ", ";
        }

        os << '"' << conf.TvmServiceIds[i] << '"';
    }
    os << ']';

    os << '\n'
       << "CreatedAt: " << conf.CreatedAt << '\n'
       << "UpdatedAt: " << conf.UpdatedAt << '\n'
       << "CreatedBy: " << conf.CreatedBy << '\n'
       << "UpdatedBy: " << conf.UpdatedBy << '\n'
       << "Version: " << conf.Version << '\n'
    ;

    return os;
}

} // namespace NSolomon::NDb::NModel
