#include "pull_settings.h"
#include "json.h"

#include <util/string/ascii.h>
#include <util/string/cast.h>
#include <util/string/strip.h>

#include <ostream>

namespace NSolomon::NDb::NModel {
namespace {

template <typename TStream>
void Write(TStream& os, const TPullSettings& settings) {
    os << "Port: " << settings.Port
        << ", Path: \'" << settings.Path << '\''
        << ", AddTsArgs: " << settings.AddTsArgs
        << ", Protocol: " << ToString(settings.Protocol)
        << ", TvmDestId: " << settings.TvmDestId
        << ", HostLabelPolicy: " << ToString(settings.HostLabelPolicy);
}

} // namespace

std::optional<TPullSettings> TPullSettings::FromJsonStr(TStringBuf data) {
    return NModel::FromJsonStr<TPullSettings>(data);
}

TPullSettings TPullSettings::FromJsonVal(const NJson::TJsonValue& json) {
    TPullSettings settings;
    settings.Port = static_cast<ui16>(Min<ui64>(json["port"].GetUInteger(), Max<ui16>()));
    settings.AddTsArgs = json["addTsArgs"].GetBoolean();

    const TString& pathStr = json["path"].GetString();
    if (Y_LIKELY(!pathStr.empty())) {
        if (Y_LIKELY(!IsAsciiSpace(pathStr.back()))) {
            settings.Path = pathStr;
        } else {
            Strip(pathStr, settings.Path);
        }
    } else {
        settings.Path = {};
    }

    EPullProtocol protocol;
    if (const auto& str = json["protocol"].GetString(); !str.empty() && TryFromString(str, protocol)) {
        settings.Protocol = protocol;
    }

    ui32 tvmDestId;
    if (const auto& str = json["tvmDestinationId"].GetString(); !str.empty() && TryFromString(str, tvmDestId)) {
        settings.TvmDestId = tvmDestId;
    }

    EHostLabelPolicy hostLabelPolicy;
    if (const auto& str = json["hostLabelPolicy"].GetString(); !str.empty() && TryFromString(str, hostLabelPolicy)) {
        settings.HostLabelPolicy = hostLabelPolicy;
    }

    return settings;
}

TString TPullSettings::ToJsonStr() const {
    return NModel::ToJsonStr<TPullSettings>(*this);
}

void TPullSettings::ToJsonVal(NJsonWriter::TBuf& json) const {
    json.BeginObject();
    json.WriteKey("port").WriteULongLong(Port);
    json.WriteKey("path").WriteString(Path);
    json.WriteKey("addTsArgs").WriteBool(AddTsArgs);
    json.WriteKey("protocol").WriteString(ToString(Protocol));
    json.WriteKey("tvmDestinationId").WriteString(ToString(TvmDestId));
    json.WriteKey("hostLabelPolicy").WriteString(ToString(HostLabelPolicy));
    json.EndObject();
}

IOutputStream& operator<<(IOutputStream& os, const TPullSettings& settings) {
    Write(os, settings);
    return os;
}

std::ostream& operator<<(std::ostream& os, const TPullSettings& settings) {
    Write(os, settings);
    return os;
}

} // namespace NSolomon::NDb::NModel
