#include "service_metric_conf.h"

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/writer/json.h>

namespace NSolomon::NDb::NModel {

TMetricConf TMetricConf::FromJsonValue(const NJson::TJsonValue& jsonValue) {
    auto&& aggrRules = jsonValue["aggrRules"].GetArray();
    TVector<TAggrRuleItem> items;
    items.reserve(aggrRules.ysize());

    for (auto&& rule: aggrRules) {
        TAggrRuleItem currentItem;
        auto&& cond = rule["cond"].GetArray();
        auto&& target = rule["target"].GetArray();

        currentItem.Cond.reserve(cond.ysize());
        currentItem.Target.reserve(target.ysize());

        for (auto&& i: cond) {
            currentItem.Cond.push_back(i.GetString());
        }

        for (auto&& i: target) {
            currentItem.Target.push_back(i.GetString());
        }

        items.push_back(std::move(currentItem));
    }

    return TMetricConf{
            .AggrRules = std::move(items),
            .RawDataMemOnly = jsonValue["rawDataMemOnly"].GetBoolean(),
    };
}

TMetricConf TMetricConf::FromString(const TString& data) {
    NJson::TJsonReaderConfig readerConfig;
    readerConfig.DontValidateUtf8 = true;

    NJson::TJsonValue jsonValue;

    try {
        NJson::ReadJsonTree(data, &readerConfig, &jsonValue, true);
    } catch (NJson::TJsonException ex) {
        Cerr << ex.AsStrBuf();
        return TMetricConf{.RawDataMemOnly = false};
    }

    return FromJsonValue(jsonValue);
}

void TMetricConf::AddToJsonOutput(NJsonWriter::TBuf& jsonOutput) const {
    jsonOutput
            .WriteKey("rawDataMemOnly").WriteBool(RawDataMemOnly)
            .WriteKey("aggrRules").BeginList();

    for (auto&& item: AggrRules) {
        jsonOutput.BeginObject()
                .WriteKey("cond").BeginList();

        for (auto&& i: item.Cond) {
            jsonOutput.WriteString(i);
        }

        jsonOutput.EndList()
                .WriteKey("target").BeginList();

        for (auto&& i: item.Target) {
            jsonOutput.WriteString(i);
        }
        jsonOutput.EndList()
                .EndObject();
    }

    jsonOutput.EndList();
}

TString TMetricConf::ToString() const {
    NJsonWriter::TBuf jsonOutput;

    jsonOutput.BeginObject();
    AddToJsonOutput(jsonOutput);
    jsonOutput.EndObject();

    return jsonOutput.Str();
}

bool operator==(const TMetricConf::TAggrRuleItem& lhs, const TMetricConf::TAggrRuleItem& rhs) {
    return lhs.Cond == rhs.Cond &&
            lhs.Target == rhs.Target;
}

bool operator==(const TMetricConf& lhs, const TMetricConf& rhs) {
    return lhs.RawDataMemOnly == rhs.RawDataMemOnly &&
            lhs.AggrRules == rhs.AggrRules;
}


} // namespace NSolomon::NDb::NModel
