#include "shard_settings.h"
#include "json.h"

#include <ostream>

namespace NSolomon::NDb::NModel {
namespace {

template <typename TStream>
void Write(TStream& os, const TShardSettings& settings) {
    os << "RetentionPolicy: " << ToString(settings.RetentionPolicy);

    if (auto* pull = std::get_if<TPullSettings>(&settings.PullOrPush)) {
        os << ", Pull: {" << *pull << '}';
    } else if (std::holds_alternative<TPushSettings>(settings.PullOrPush)) {
        os << ", Push: {}";
    }

    os << ", AggregationSettings: {" << settings.AggregationSettings << '}';
    os << ", Grid: " << ToString(settings.Grid);
    os << ", Interval: " << ToString(settings.Interval);
}

} // namespace

std::optional<TShardSettings> TShardSettings::FromJsonStr(TStringBuf data) {
    return NModel::FromJsonStr<TShardSettings>(data);
}

TShardSettings TShardSettings::FromJsonVal(const NJson::TJsonValue& json) {
    TShardSettings setts;

    if (const TString& type = json["type"].GetString(); type == "PUSH") {
        setts.PullOrPush = TPushSettings{};
    } else if (type == "PULL" && json.Has("pullSettings")) {
        setts.PullOrPush = TPullSettings::FromJsonVal(json["pullSettings"]);
    }

    setts.AggregationSettings = TAggregationSettings::FromJsonVal(json["aggregationSettings"]);

    ERetentionPolicy retentionPolicy;
    if (const auto& str = json["retentionPolicy"].GetString(); !str.empty() && TryFromString(str, retentionPolicy)) {
        setts.RetentionPolicy = retentionPolicy;
    }

    setts.Grid = TDuration::Seconds(json["grid"].GetUInteger());
    setts.Interval = TDuration::Seconds(json["interval"].GetUInteger());
    setts.MetricsTtl = TDuration::Days(json["metricsTtl"].GetUInteger());

    return setts;
}

TString TShardSettings::ToJsonStr() const {
    return NModel::ToJsonStr<TShardSettings>(*this);
}

void TShardSettings::ToJsonVal(NJsonWriter::TBuf& json) const {
    json.BeginObject();

    if (auto* pull = std::get_if<TPullSettings>(&PullOrPush)) {
        json.WriteKey("type").WriteString("PULL");
        json.WriteKey("pullSettings");
        pull->ToJsonVal(json);
    } else if (std::holds_alternative<TPushSettings>(PullOrPush)){
        json.WriteKey("type").WriteString("PUSH");
    }

    json.WriteKey("aggregationSettings");
    AggregationSettings.ToJsonVal(json);

    json.WriteKey("retentionPolicy").WriteString(ToString(RetentionPolicy));
    json.WriteKey("grid").WriteULongLong(Grid.Seconds());
    json.WriteKey("interval").WriteULongLong(Interval.Seconds());
    json.WriteKey("metricsTtl").WriteULongLong(MetricsTtl.Days());
    json.EndObject();
}

IOutputStream& operator<<(IOutputStream& os, const TShardSettings& settings) {
    Write(os, settings);
    return os;
}

std::ostream& operator<<(std::ostream& os, const TShardSettings& settings) {
    Write(os, settings);
    return os;
}

} // namespace NSolomon::NDb::NModel
