#pragma once

#include "aggregation_settings.h"
#include "pull_settings.h"
#include "push_settings.h"

#include <library/cpp/json/writer/json.h>
#include <library/cpp/json/writer/json_value.h>

#include <util/datetime/base.h>
#include <util/ysaveload.h>

#include <variant>
#include <optional>

namespace NSolomon::NDb::NModel {

enum class ERetentionPolicy {
    UNDEFINED,
    FIVE_MIN_AFTER_WEEK,
    ONE_MIN_AFTER_MONTH_FIVE_MIN_AFTER_THREE_MONTHS,
    ONE_MIN_AFTER_MONTH_FIVE_MIN_AFTER_TWO_MONTHS,
    FIVE_MIN_AFTER_TWO_MONTHS,
    KEEP_FOREVER,
};

inline void Save(IOutputStream* s, const std::variant<std::monostate, TPullSettings, TPushSettings>& pullOrPush) {
    ::Save(s, pullOrPush.index());
    if (auto* pull = std::get_if<TPullSettings>(&pullOrPush)) {
        ::Save(s, *pull);
    }

    // TPushSettings is empty right now
    static_assert(sizeof(TPushSettings) == 1);
}

inline void Load(IInputStream* s, std::variant<std::monostate, TPullSettings, TPushSettings>& pullOrPush) {
    size_t index;
    ::Load(s, index);

    Y_ASSERT(index <= 2);

    if (index == 0) {
        pullOrPush = std::monostate{};
    } else if (index == 1) {
        TPullSettings pullSettings;
        ::Load(s, pullSettings);
        pullOrPush.emplace<TPullSettings>(std::move(pullSettings));
    } else {
        pullOrPush = TPushSettings{};
    }
}

struct TShardSettings {
    std::variant<std::monostate, TPullSettings, TPushSettings> PullOrPush;
    TAggregationSettings AggregationSettings;
    ERetentionPolicy RetentionPolicy{ERetentionPolicy::UNDEFINED};
    TDuration Grid;
    TDuration Interval;
    TDuration MetricsTtl;

    Y_SAVELOAD_DEFINE(PullOrPush, AggregationSettings, RetentionPolicy, Grid, Interval, MetricsTtl);

    static std::optional<TShardSettings> FromJsonStr(TStringBuf data);
    static TShardSettings FromJsonVal(const NJson::TJsonValue& json);

    TString ToJsonStr() const;
    void ToJsonVal(NJsonWriter::TBuf& json) const;

    bool operator==(const TShardSettings&) const = default;
};

IOutputStream& operator<<(IOutputStream& os, const TShardSettings& settings);
std::ostream& operator<<(std::ostream& os, const TShardSettings& settings);

} // namespace NSolomon::NDb::NModel
