#include <solomon/libs/cpp/conf_db/model/pull_settings.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <sstream>

using namespace NSolomon::NDb::NModel;

TEST(TPullSettingsTest, ArcadiaStream) {
    {
        TStringStream ss;
        ss << TPullSettings{};
        ASSERT_EQ(ss.Str(), "Port: 0, Path: '', AddTsArgs: 0, Protocol: UNKNOWN, TvmDestId: 0, HostLabelPolicy: UNKNOWN");
    }
    {
        TStringStream ss;
        ss << TPullSettings{
            .Port = 1337,
            .Path = "/some/path",
            .AddTsArgs = true,
            .Protocol = EPullProtocol::HTTPS,
            .TvmDestId = 12345678,
            .HostLabelPolicy = EHostLabelPolicy::FULL_HOSTNAME,
        };
        ASSERT_EQ(ss.Str(), "Port: 1337, Path: '/some/path', AddTsArgs: 1, Protocol: HTTPS, TvmDestId: 12345678, HostLabelPolicy: FULL_HOSTNAME");
    }
}

TEST(TPullSettingsTest, StdStream) {
    {
        std::stringstream ss;
        ss << TPullSettings{};
        ASSERT_EQ(ss.str(), "Port: 0, Path: '', AddTsArgs: 0, Protocol: UNKNOWN, TvmDestId: 0, HostLabelPolicy: UNKNOWN");
    }
    {
        std::stringstream ss;
        ss << TPullSettings{
            .Port = 1337,
            .Path = "/some/path",
            .AddTsArgs = true,
            .Protocol = EPullProtocol::HTTPS,
            .TvmDestId = 12345678,
            .HostLabelPolicy = EHostLabelPolicy::FULL_HOSTNAME,
        };
        ASSERT_EQ(ss.str(), "Port: 1337, Path: '/some/path', AddTsArgs: 1, Protocol: HTTPS, TvmDestId: 12345678, HostLabelPolicy: FULL_HOSTNAME");
    }
}

TEST(TPullSettingsTest, FromJson) {
    TString json = R"({
        "port": 3001,
        "path": "/some/path",
        "addTsArgs": true,
        "protocol": "HTTPS",
        "tvmDestinationId": "12345678",
        "hostLabelPolicy": "FULL_HOSTNAME"
    })";

    auto s = TPullSettings::FromJsonStr(json);
    ASSERT_TRUE(s.has_value());

    EXPECT_EQ(s->Port, 3001u);
    EXPECT_EQ(s->Path, "/some/path");
    EXPECT_TRUE(s->AddTsArgs);
    EXPECT_EQ(s->Protocol, EPullProtocol::HTTPS);
    EXPECT_EQ(s->TvmDestId, 12345678u);
    EXPECT_EQ(s->HostLabelPolicy, EHostLabelPolicy::FULL_HOSTNAME);
}

TEST(TPullSettingsTest, FromJson_AbsentPort) {
    TString json = R"({
        "path": "/some/path",
        "addTsArgs": true,
        "protocol": "HTTPS",
        "tvmDestinationId": "12345678",
        "hostLabelPolicy": "FULL_HOSTNAME"
    })";

    auto s = TPullSettings::FromJsonStr(json);
    ASSERT_TRUE(s.has_value());

    EXPECT_EQ(s->Port, 0u);
    EXPECT_EQ(s->Path, "/some/path");
    EXPECT_TRUE(s->AddTsArgs);
    EXPECT_EQ(s->Protocol, EPullProtocol::HTTPS);
    EXPECT_EQ(s->TvmDestId, 12345678u);
    EXPECT_EQ(s->HostLabelPolicy, EHostLabelPolicy::FULL_HOSTNAME);
}

TEST(TPullSettingsTest, FromJson_InvalidPort) {
    TString json = R"({
        "port": "some_string",
        "path": "/some/path",
        "addTsArgs": true,
        "protocol": "HTTPS",
        "tvmDestinationId": "12345678",
        "hostLabelPolicy": "FULL_HOSTNAME"
    })";

    auto s = TPullSettings::FromJsonStr(json);
    ASSERT_TRUE(s.has_value());

    EXPECT_EQ(s->Port, 0u);
    EXPECT_EQ(s->Path, "/some/path");
    EXPECT_TRUE(s->AddTsArgs);
    EXPECT_EQ(s->Protocol, EPullProtocol::HTTPS);
    EXPECT_EQ(s->TvmDestId, 12345678u);
    EXPECT_EQ(s->HostLabelPolicy, EHostLabelPolicy::FULL_HOSTNAME);
}

TEST(TPullSettingsTest, FromJson_AbsentPath) {
    TString json = R"({
        "port": 3001,
        "addTsArgs": true,
        "protocol": "HTTPS",
        "tvmDestinationId": "12345678",
        "hostLabelPolicy": "FULL_HOSTNAME"
    })";

    auto s = TPullSettings::FromJsonStr(json);
    ASSERT_TRUE(s.has_value());

    EXPECT_EQ(s->Port, 3001u);
    EXPECT_EQ(s->Path, "");
    EXPECT_TRUE(s->AddTsArgs);
    EXPECT_EQ(s->Protocol, EPullProtocol::HTTPS);
    EXPECT_EQ(s->TvmDestId, 12345678u);
    EXPECT_EQ(s->HostLabelPolicy, EHostLabelPolicy::FULL_HOSTNAME);
}

TEST(TPullSettingsTest, FromJson_PathWithTrailingWhitespaces) {
    {
        TString json = R"({
            "port": 3001,
            "path": "/some/path?key=value ",
            "addTsArgs": false,
            "protocol": "HTTPS"
        })";

        auto s = TPullSettings::FromJsonStr(json);
        ASSERT_TRUE(s.has_value());

        EXPECT_EQ(s->Port, 3001u);
        EXPECT_EQ(s->Path, "/some/path?key=value");
        EXPECT_FALSE(s->AddTsArgs);
    }

    {
        TString json = R"({
            "port": 3001,
            "path": "/some/path?key=value\n\t",
            "addTsArgs": false,
            "protocol": "HTTPS"
        })";

        auto s = TPullSettings::FromJsonStr(json);
        ASSERT_TRUE(s.has_value());

        EXPECT_EQ(s->Port, 3001u);
        EXPECT_EQ(s->Path, "/some/path?key=value");
        EXPECT_FALSE(s->AddTsArgs);
    }
}

TEST(TPullSettingsTest, FromJson_AbsentAddTsArgs) {
    TString json = R"({
        "port": 3001,
        "path": "/some/path",
        "protocol": "HTTPS",
        "tvmDestinationId": "12345678",
        "hostLabelPolicy": "FULL_HOSTNAME"
    })";

    auto s = TPullSettings::FromJsonStr(json);
    ASSERT_TRUE(s.has_value());

    EXPECT_EQ(s->Port, 3001u);
    EXPECT_EQ(s->Path, "/some/path");
    EXPECT_FALSE(s->AddTsArgs);
    EXPECT_EQ(s->Protocol, EPullProtocol::HTTPS);
    EXPECT_EQ(s->TvmDestId, 12345678u);
    EXPECT_EQ(s->HostLabelPolicy, EHostLabelPolicy::FULL_HOSTNAME);
}

TEST(TPullSettingsTest, FromJson_InvalidAddTsArgs) {
    TString json = R"({
        "port": 3001,
        "path": "/some/path",
        "addTsArgs": 123,
        "protocol": "HTTPS",
        "tvmDestinationId": "12345678",
        "hostLabelPolicy": "FULL_HOSTNAME"
    })";

    auto s = TPullSettings::FromJsonStr(json);
    ASSERT_TRUE(s.has_value());

    EXPECT_EQ(s->Port, 3001u);
    EXPECT_EQ(s->Path, "/some/path");
    EXPECT_FALSE(s->AddTsArgs);
    EXPECT_EQ(s->Protocol, EPullProtocol::HTTPS);
    EXPECT_EQ(s->TvmDestId, 12345678u);
    EXPECT_EQ(s->HostLabelPolicy, EHostLabelPolicy::FULL_HOSTNAME);
}

TEST(TPullSettingsTest, FromJson_AbsentProtocol) {
    TString json = R"({
        "port": 3001,
        "path": "/some/path",
        "addTsArgs": true,
        "tvmDestinationId": "12345678",
        "hostLabelPolicy": "FULL_HOSTNAME"
    })";

    auto s = TPullSettings::FromJsonStr(json);
    ASSERT_TRUE(s.has_value());

    EXPECT_EQ(s->Port, 3001u);
    EXPECT_EQ(s->Path, "/some/path");
    EXPECT_TRUE(s->AddTsArgs);
    EXPECT_EQ(s->Protocol, EPullProtocol::UNKNOWN);
    EXPECT_EQ(s->TvmDestId, 12345678u);
    EXPECT_EQ(s->HostLabelPolicy, EHostLabelPolicy::FULL_HOSTNAME);
}

TEST(TPullSettingsTest, FromJson_InvalidProtocol) {
    TString json = R"({
        "port": 3001,
        "path": "/some/path",
        "addTsArgs": true,
        "protocol": "GOPHER",
        "tvmDestinationId": "12345678",
        "hostLabelPolicy": "FULL_HOSTNAME"
    })";

    auto s = TPullSettings::FromJsonStr(json);
    ASSERT_TRUE(s.has_value());

    EXPECT_EQ(s->Port, 3001u);
    EXPECT_EQ(s->Path, "/some/path");
    EXPECT_TRUE(s->AddTsArgs);
    EXPECT_EQ(s->Protocol, EPullProtocol::UNKNOWN);
    EXPECT_EQ(s->TvmDestId, 12345678u);
    EXPECT_EQ(s->HostLabelPolicy, EHostLabelPolicy::FULL_HOSTNAME);
}

TEST(TPullSettingsTest, FromJson_AbsentTvmDestinationId) {
    TString json = R"({
        "port": 3001,
        "path": "/some/path",
        "addTsArgs": true,
        "protocol": "HTTPS",
        "hostLabelPolicy": "FULL_HOSTNAME"
    })";

    auto s = TPullSettings::FromJsonStr(json);
    ASSERT_TRUE(s.has_value());

    EXPECT_EQ(s->Port, 3001u);
    EXPECT_EQ(s->Path, "/some/path");
    EXPECT_TRUE(s->AddTsArgs);
    EXPECT_EQ(s->Protocol, EPullProtocol::HTTPS);
    EXPECT_EQ(s->TvmDestId, 0u);
    EXPECT_EQ(s->HostLabelPolicy, EHostLabelPolicy::FULL_HOSTNAME);
}

TEST(TPullSettingsTest, FromJson_InvalidTvmDestinationId) {
    TString json = R"({
        "port": 3001,
        "path": "/some/path",
        "addTsArgs": true,
        "protocol": "HTTPS",
        "tvmDestinationId": "one-two-three",
        "hostLabelPolicy": "FULL_HOSTNAME"
    })";

    auto s = TPullSettings::FromJsonStr(json);
    ASSERT_TRUE(s.has_value());

    EXPECT_EQ(s->Port, 3001u);
    EXPECT_EQ(s->Path, "/some/path");
    EXPECT_TRUE(s->AddTsArgs);
    EXPECT_EQ(s->Protocol, EPullProtocol::HTTPS);
    EXPECT_EQ(s->TvmDestId, 0u);
    EXPECT_EQ(s->HostLabelPolicy, EHostLabelPolicy::FULL_HOSTNAME);
}

TEST(TPullSettingsTest, FromJson_AbsentHostLabelPolicy) {
    TString json = R"({
        "port": 3001,
        "path": "/some/path",
        "addTsArgs": true,
        "protocol": "HTTPS",
        "tvmDestinationId": "12345678"
    })";

    auto s = TPullSettings::FromJsonStr(json);
    ASSERT_TRUE(s.has_value());

    EXPECT_EQ(s->Port, 3001u);
    EXPECT_EQ(s->Path, "/some/path");
    EXPECT_TRUE(s->AddTsArgs);
    EXPECT_EQ(s->Protocol, EPullProtocol::HTTPS);
    EXPECT_EQ(s->TvmDestId, 12345678u);
    EXPECT_EQ(s->HostLabelPolicy, EHostLabelPolicy::UNKNOWN);
}

TEST(TPullSettingsTest, FromJson_InvalidHostLabelPolicy) {
    TString json = R"({
        "port": 3001,
        "path": "/some/path",
        "addTsArgs": true,
        "protocol": "HTTPS",
        "tvmDestinationId": "12345678",
        "hostLabelPolicy": "SOLOMON"
    })";

    auto s = TPullSettings::FromJsonStr(json);
    ASSERT_TRUE(s.has_value());

    EXPECT_EQ(s->Port, 3001u);
    EXPECT_EQ(s->Path, "/some/path");
    EXPECT_TRUE(s->AddTsArgs);
    EXPECT_EQ(s->Protocol, EPullProtocol::HTTPS);
    EXPECT_EQ(s->TvmDestId, 12345678u);
    EXPECT_EQ(s->HostLabelPolicy, EHostLabelPolicy::UNKNOWN);
}

TEST(TPullSettingsTest, FromJson_InvalidJson) {
    TString json = R"({
        "port":
        "path":
    })";

    auto s = TPullSettings::FromJsonStr(json);
    ASSERT_FALSE(s.has_value());
}

TEST(TPullSettingsTest, ToJson) {
    TPullSettings settings{
        .Port = 1337,
        .Path = "/some/path",
        .AddTsArgs = true,
        .Protocol = EPullProtocol::HTTPS,
        .TvmDestId = 12345678,
        .HostLabelPolicy = EHostLabelPolicy::FULL_HOSTNAME,
    };

    auto json = settings.ToJsonStr();
    ASSERT_EQ(json, R"({"port":1337,"path":"/some/path","addTsArgs":true,"protocol":"HTTPS","tvmDestinationId":"12345678","hostLabelPolicy":"FULL_HOSTNAME"})");
}
