#include <solomon/libs/cpp/conf_db/model/shard_settings.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <sstream>

using namespace NSolomon::NDb::NModel;

TEST(TShardSettingsTest, FromJson_WithPullSettings) {
    TString json = R"({
        "type": "PULL",
        "pullSettings": {
            "port": 8080,
            "path": "/some/path",
            "addTsArgs": true,
            "tvmDestinationId": "12345678",
            "hostLabelPolicy": "FULL_HOSTNAME"
        },
        "grid": 15,
        "retentionPolicy": "FIVE_MIN_AFTER_WEEK",
        "aggregationSettings": {
            "enabled": true,
            "aggrRules": [
                { "cond": [ "host=*" ], "target": [ "host=cluster" ], "aggr": "SUM" },
                { "cond": [ "host=*" ], "target": [ "host={{DC}}" ], "aggr": "LAST" }
            ],
            "memOnly": true
        },
        "metricsTtl":23,
        "interval":45
    })";

    auto settings = TShardSettings::FromJsonStr(json);
    ASSERT_TRUE(settings.has_value());

    auto* pullSettings = std::get_if<TPullSettings>(&settings->PullOrPush);
    ASSERT_TRUE(pullSettings);
    EXPECT_EQ(pullSettings->Port, 8080u);
    EXPECT_EQ(pullSettings->Path, "/some/path");
    EXPECT_TRUE(pullSettings->AddTsArgs);
    EXPECT_EQ(pullSettings->TvmDestId, 12345678u);
    EXPECT_EQ(pullSettings->HostLabelPolicy, EHostLabelPolicy::FULL_HOSTNAME);

    EXPECT_EQ(settings->Grid, TDuration::Seconds(15));
    EXPECT_EQ(settings->Interval, TDuration::Seconds(45));
    EXPECT_EQ(settings->RetentionPolicy, ERetentionPolicy::FIVE_MIN_AFTER_WEEK);
    EXPECT_EQ(settings->MetricsTtl, TDuration::Days(23));

    const auto& aggrSettings = settings->AggregationSettings;
    EXPECT_TRUE(aggrSettings.Enabled);
    EXPECT_TRUE(aggrSettings.MemOnly);

    ASSERT_EQ(aggrSettings.Rules.size(), 2u);

    const auto& rule1 = aggrSettings.Rules[0];
    EXPECT_EQ(rule1.Function, EAggregationFunction::SUM);
    EXPECT_EQ(rule1.Conditions, std::vector<TString>{"host=*"});
    EXPECT_EQ(rule1.Targets, std::vector<TString>{"host=cluster"});

    const auto& rule2 = aggrSettings.Rules[1];
    EXPECT_EQ(rule2.Function, EAggregationFunction::LAST);
    EXPECT_EQ(rule2.Conditions, std::vector<TString>{"host=*"});
    EXPECT_EQ(rule2.Targets, std::vector<TString>{"host={{DC}}"});
}

TEST(TShardSettingsTest, FromJson_WithPushSettings) {
    TString json = R"({
        "type": "PUSH",
        "grid": 15,
        "interval": 45,
        "retentionPolicy": "KEEP_FOREVER"
    })";

    auto settings = TShardSettings::FromJsonStr(json);
    ASSERT_TRUE(settings.has_value());

    ASSERT_TRUE(std::holds_alternative<TPushSettings>(settings->PullOrPush));

    EXPECT_EQ(settings->Grid, TDuration::Seconds(15));
    EXPECT_EQ(settings->Interval, TDuration::Seconds(45));
    EXPECT_EQ(settings->RetentionPolicy, ERetentionPolicy::KEEP_FOREVER);

    const auto& aggrSettings = settings->AggregationSettings;
    EXPECT_FALSE(aggrSettings.Enabled);
    EXPECT_FALSE(aggrSettings.MemOnly);
    EXPECT_TRUE(aggrSettings.Rules.empty());
}

TEST(TShardSettingsTest, ToJson) {
    TShardSettings settings{
        .PullOrPush = TPullSettings{
            .Port = 8080,
            .Path = "/metrics/json",
            .AddTsArgs = true,
            .Protocol = EPullProtocol::HTTPS,
            .TvmDestId = 1234567,
            .HostLabelPolicy = EHostLabelPolicy::NO_HOSTNAME,
        },
        .AggregationSettings = {
            .Rules = {
                {
                    .Conditions = {"host=*"},
                    .Targets = {"host=cluster"},
                    .Function = EAggregationFunction::LAST
                }
            },
            .Enabled = true,
            .MemOnly = false,
        },
        .RetentionPolicy = ERetentionPolicy::KEEP_FOREVER,
        .Grid = TDuration::Seconds(30),
        .Interval = TDuration::Seconds(60),
        .MetricsTtl = TDuration::Days(23),
    };

    auto json = settings.ToJsonStr();
    ASSERT_EQ(json, "{"
        "\"type\":\"PULL\","
        "\"pullSettings\":{"
            "\"port\":8080,"
            "\"path\":\"/metrics/json\","
            "\"addTsArgs\":true,"
            "\"protocol\":\"HTTPS\","
            "\"tvmDestinationId\":\"1234567\","
            "\"hostLabelPolicy\":\"NO_HOSTNAME\""
        "},"
        "\"aggregationSettings\":{"
            "\"aggrRules\":["
                "{\"aggr\":\"LAST\",\"cond\":[\"host=*\"],\"target\":[\"host=cluster\"]}"
            "],"
            "\"enabled\":true,"
            "\"memOnly\":false"
        "},"
        "\"retentionPolicy\":\"KEEP_FOREVER\","
        "\"grid\":30,"
        "\"interval\":60,"
        "\"metricsTtl\":23"
    "}");
}

TEST(TShardSettingsTest, ToJson_Default) {
    TShardSettings settings;
    auto json = settings.ToJsonStr();

    ASSERT_EQ(json, R"({"aggregationSettings":{"aggrRules":[],"enabled":false,"memOnly":false},"retentionPolicy":"UNDEFINED","grid":0,"interval":0,"metricsTtl":0})");
}

TEST(TShardSettingsTest, Equality) {
    {
        TShardSettings a;
        TShardSettings b;
        ASSERT_EQ(a, b);
    }
    {
        TShardSettings a{
            .PullOrPush = TPushSettings{}
        };
        TShardSettings b;
        ASSERT_NE(a, b);
    }
    {
        TShardSettings a{
            .PullOrPush = TPushSettings{}
        };
        TShardSettings b{
            .PullOrPush = TPushSettings{}
        };
        ASSERT_EQ(a, b);
    }
    {
        TShardSettings a{
            .PullOrPush = TPullSettings{}
        };
        TShardSettings b;
        ASSERT_NE(a, b);
    }
    {
        TShardSettings a{
            .PullOrPush = TPullSettings{
                .Protocol = EPullProtocol::HTTP
            }
        };
        TShardSettings b{
            .PullOrPush = TPullSettings{
                .Protocol = EPullProtocol::HTTP
            }
        };
        ASSERT_EQ(a, b);
    }
    {
        TShardSettings a{
            .PullOrPush = TPullSettings{
                .Protocol = EPullProtocol::HTTP
            }
        };
        TShardSettings b{
            .PullOrPush = TPullSettings{
                .Protocol = EPullProtocol::HTTPS
            }
        };
        ASSERT_NE(a, b);
    }
}
