#include "puller.h"

#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/backoff/backoff.h>
#include <solomon/libs/cpp/backoff/jitter.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/containers/absl_flat_hash/flat_hash_set.h>

#include <util/generic/scope.h>
#include <util/stream/file.h>
#include <util/system/fs.h>

namespace NSolomon {
namespace {

using namespace NActors;

const TString CONFIGS_PATH = "/Berkanavt/solomon/config_db/configs";

class TConfigsPuller: public TActorBootstrapped<TConfigsPuller>, TPrivateEvents {
    enum {
        LoadConfigs = SpaceBegin,
        End,
    };
    static_assert(End < SpaceEnd, "too many events");

    struct TLoadConfigs: public NActors::TEventLocal<TLoadConfigs, LoadConfigs> {
    };

public:
    explicit TConfigsPuller(TConfigsPullerOptions opts)
        : ProjectDao_{std::move(opts.ProjectDao)}
        , ClusterDao_{std::move(opts.ClusterDao)}
        , ServiceDao_{std::move(opts.ServiceDao)}
        , ShardDao_{std::move(opts.ShardDao)}
        , AgentDao_{std::move(opts.AgentDao)}
        , ProviderDao_{std::move(opts.ProviderDao)}
        , UpdateBackoff_{opts.UpdateInterval, opts.MaxUpdateInterval}
        , Registry_{opts.Registry}
        , Configs_{std::move(opts.ConfigsForTesting)}
        , IsInTestingMode_{!Configs_.Empty()}
    {
        if (!IsInTestingMode_ && NFs::Exists(CONFIGS_PATH)) {
            try {
                TFileInput input(CONFIGS_PATH);
                Load(&input, Configs_);
            } catch (const yexception& e) {
                Cerr << "exception while reading configs: " << e.what();
                Configs_ = {};
            }
        }
    }

    void Bootstrap() {
        Become(&TThis::Main);
        ScheduleConfigsUpdate();
    }

    STATEFN(Main) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NTableLoader::TEvProjectsReady, OnSomethingReady<NTableLoader::TEvProjectsReady>);
            hFunc(NTableLoader::TEvClustersReady, OnSomethingReady<NTableLoader::TEvClustersReady>);
            hFunc(NTableLoader::TEvServicesReady, OnSomethingReady<NTableLoader::TEvServicesReady>);
            hFunc(NTableLoader::TEvShardsReady, OnSomethingReady<NTableLoader::TEvShardsReady>);
            hFunc(NTableLoader::TEvAgentsReady, OnSomethingReady<NTableLoader::TEvAgentsReady>);
            hFunc(NTableLoader::TEvProvidersReady, OnSomethingReady<NTableLoader::TEvProvidersReady>);
            hFunc(TConfigsPullerEvents::TSubscribe, OnSubscribe);
            sFunc(TLoadConfigs, LoadConfigsFromDB);
            sFunc(TEvents::TEvPoison, PassAway);
        }
    }

private:
    void ScheduleConfigsUpdate() {
        if (IsInTestingMode_) {
            return;
        }

        auto delay = UpdateBackoff_();
        Schedule(delay, new TLoadConfigs);

        MON_INFO(ConfigsPuller, "will fetch configs after: " << delay);
    }

    void OnSubscribe(const TConfigsPullerEvents::TSubscribe::TPtr& evPtr) {
        Subscribers_.emplace(evPtr->Sender);

        if (!Configs_.Empty()) {
            Send(evPtr->Sender, new TConfigsPullerEvents::TConfigsResponse{Configs_});
        }
    }

    void LoadConfigsFromDB() {
        if (ProjectDao_) {
            Register(NTableLoader::CreateProjectLoaderActor(ProjectDao_, {Registry_}, SelfId()));
            ++Inflight_;
        }

        if (ClusterDao_) {
            Register(NTableLoader::CreateClusterLoaderActor(ClusterDao_, {Registry_}, SelfId()));
            ++Inflight_;
        }

        if (ServiceDao_) {
            Register(NTableLoader::CreateServiceLoaderActor(ServiceDao_, {Registry_}, SelfId()));
            ++Inflight_;
        }

        if (ShardDao_) {
            Register(NTableLoader::CreateShardLoaderActor(ShardDao_, {Registry_}, SelfId()));
            ++Inflight_;
        }

        if (AgentDao_) {
            Register(NTableLoader::CreateAgentsLoaderActor(AgentDao_, {Registry_}, SelfId()));
            ++Inflight_;
        }

        if (ProviderDao_) {
            Register(NTableLoader::CreateProvidersLoaderActor(ProviderDao_, {Registry_}, SelfId()));
            ++Inflight_;
        }

        MON_DEBUG(ConfigsPuller, "sent config load requests, inflight = " << Inflight_);
    }

    void UpdateConfigs(TVector<NDb::NModel::TProjectConfig>&& projects) {
        MON_INFO(ConfigsPuller, "Load " << projects.size() << " projects from ConfDB");
        NewConfigs_.Projects = std::move(projects);
    }

    void UpdateConfigs(TVector<NDb::NModel::TClusterConfig>&& clusters) {
        MON_INFO(ConfigsPuller, "Load " << clusters.size() << " clusters from ConfDB");
        NewConfigs_.Clusters = std::move(clusters);
    }

    void UpdateConfigs(TVector<NDb::NModel::TServiceConfig>&& services) {
        MON_INFO(ConfigsPuller, "Load " << services.size() << " services from ConfDB");
        NewConfigs_.Services = std::move(services);
    }

    void UpdateConfigs(TVector<NDb::NModel::TShardConfig>&& shards) {
        MON_INFO(ConfigsPuller, "Load " << shards.size() << " shards from ConfDB");
        NewConfigs_.Shards = std::move(shards);
    }

    void UpdateConfigs(TVector<NDb::NModel::TAgentConfig>&& agents) {
        MON_INFO(ConfigsPuller, "Load " << agents.size() << " agents from ConfDB");
        NewConfigs_.Agents = std::move(agents);
    }

    void UpdateConfigs(TVector<NDb::NModel::TProviderConfig>&& providers) {
        MON_INFO(ConfigsPuller, "Load " << providers.size() << " providers from ConfDB");
        NewConfigs_.Providers = std::move(providers);
    }

    template <typename TEv>
    void OnSomethingReady(typename TEv::TPtr& ev) {
        auto& result = ev->Get()->Result;

        if (result.Fail()) {
            MON_ERROR(ConfigsPuller, "failed to load configs: " << result.Error().Message());
            GotError_ = true;
        } else {
            UpdateConfigs(result.Extract());
        }

        --Inflight_;
        if (Inflight_ == 0) {
            OnAllReady();
        }
    }

    void OnAllReady() {
        MON_INFO(ConfigsPuller, "Got all configs, will be sent to " << Subscribers_.size() << " subscribers");

        Y_DEFER {
            ScheduleConfigsUpdate();
        };

        if (GotError_) {
            GotError_ = false;
            return;
        }

        UpdateBackoff_.Reset();
        Configs_ = std::move(NewConfigs_);
        for (const auto& sub: Subscribers_) {
            Send(sub, new TConfigsPullerEvents::TConfigsResponse{Configs_});
        }

        try {
            TFileOutput output(CONFIGS_PATH);
            Save(&output, Configs_);
        } catch (const yexception& e) {
            MON_ERROR(ConfigsPuller, "exception while writing configs: " << e);
        }
    }

private:
    NDb::IProjectConfigDaoPtr ProjectDao_;
    NDb::IClusterConfigDaoPtr ClusterDao_;
    NDb::IServiceConfigDaoPtr ServiceDao_;
    NDb::IShardConfigDaoPtr ShardDao_;
    NDb::IAgentConfigDaoPtr AgentDao_;
    NDb::IProviderConfigDaoPtr ProviderDao_;

    TLinearBackoff<THalfJitter> UpdateBackoff_;
    NMonitoring::IMetricRegistry& Registry_;
    absl::flat_hash_set<TActorId, THash<TActorId>> Subscribers_;
    ui16 Inflight_ = 0;
    bool GotError_ = false;
    TConfigs Configs_;
    TConfigs NewConfigs_;
    bool IsInTestingMode_{false};
};

} // namespace

std::unique_ptr<NActors::IActor> CreateConfigsPuller(TConfigsPullerOptions opts) {
    return std::make_unique<TConfigsPuller>(std::move(opts));
}

} // namespace NSolomon
