#include <solomon/libs/cpp/conf_db/puller/puller.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/testing/gtest/gtest.h>

using namespace NActors;
using namespace NSolomon;

class TConfigsPullerTest: public ::testing::Test {
public:
    void SetUp() override {
        ActorRuntime = TTestActorRuntime::CreateInited(1, false, true);
        Registry = std::make_shared<NMonitoring::TMetricRegistry>();
        EdgeId = ActorRuntime->AllocateEdgeActor();
    }

    void TearDown() override {
        ActorRuntime.Reset();
    }

    TActorId RegisterConfigsPuller(
            TIntrusivePtr<NSolomon::NDb::IShardConfigDao> dao,
            const TVector<NDb::NModel::TShardConfig>& configsForTesting = {})
    {
        TConfigs configs{.Shards = configsForTesting};
        return ActorRuntime->Register(CreateConfigsPuller({
                .ShardDao = std::move(dao),
                .UpdateInterval = TDuration::Seconds(10),
                .MaxUpdateInterval = TDuration::Seconds(10) + TDuration::MicroSeconds(1),
                .Registry = *Registry,
                .ConfigsForTesting = std::move(configs),
        }).release());
    }

    THolder<NSolomon::TTestActorRuntime> ActorRuntime;
    std::shared_ptr<NMonitoring::TMetricRegistry> Registry;
    TActorId EdgeId;
};

enum class EErrorPolicy {
    WithoutErrors,
    ErrorOnFirstCall,
};

class TShardDao: public NSolomon::NDb::IShardConfigDao {
public:
    explicit TShardDao(EErrorPolicy errorPolicy)
        : ErrorPolicy_{errorPolicy}
        , Shards{
            {
                .Id = "shard1",
            },
            {
                .Id = "shard2",
            },
        }
    {
    }

private:
    NDb::TAsyncVoid CreateTable() override {
        return NSolomon::NDb::TAsyncVoid();
    }

    NDb::TAsyncVoid DropTable() override {
        return NSolomon::NDb::TAsyncVoid();
    }

    NDb::TAsyncVoid Insert(const NDb::NModel::TShardConfig&) override {
        return NSolomon::NDb::TAsyncVoid();
    }

    NDb::TAsyncVoid Delete(const TString&, const TString&) override {
        return NSolomon::NDb::TAsyncVoid();
    }

    NDb::TAsyncShardConfigs GetAll() override {
        ++GetAllCnt;
        auto p = NThreading::NewPromise<NSolomon::NDb::TAsyncShardConfigs::value_type>();

        if (GetAllCnt == 1 && ErrorPolicy_ == EErrorPolicy::ErrorOnFirstCall) {
            p.SetException("error");
        } else {
            p.SetValue(Shards);
        }

        return p.GetFuture();
    }

    NDb::TAsyncShardConfig GetById(const TString&, const TString&) override {
        return NSolomon::NDb::TAsyncShardConfig();
    }

private:
    EErrorPolicy ErrorPolicy_;

public:
    size_t GetAllCnt{0};
    TVector<NDb::NModel::TShardConfig> Shards;
};

TEST_F(TConfigsPullerTest, HappyPath) {
    auto dao = MakeIntrusive<TShardDao>(EErrorPolicy::WithoutErrors);
    auto cp = RegisterConfigsPuller(dao);
    ActorRuntime->WaitForBootstrap();

    ActorRuntime->Send(cp, EdgeId, MakeHolder<TConfigsPullerEvents::TSubscribe>());

    auto pull = [&]() {
        ActorRuntime->AdvanceCurrentTime(TDuration::Seconds(11));
        ActorRuntime->DispatchEvents({}, TDuration::Seconds(1));

        return ActorRuntime->GrabEdgeEvent<TConfigsPullerEvents::TConfigsResponse>(EdgeId);
    };

    size_t callCnt;
    {
        auto resp = pull();
        ASSERT_TRUE(resp);

        EXPECT_GE(dao->GetAllCnt, 1ul);
        callCnt = dao->GetAllCnt;
        EXPECT_EQ(dao->Shards.size(), resp->Get()->Configs.Shards.size());
        EXPECT_EQ(dao->Shards, resp->Get()->Configs.Shards);
    }

    {
        auto resp = pull();
        ASSERT_TRUE(resp);

        EXPECT_GE(dao->GetAllCnt, 2ul);
        EXPECT_GT(dao->GetAllCnt, callCnt);
        EXPECT_EQ(dao->Shards.size(), resp->Get()->Configs.Shards.size());
        EXPECT_EQ(dao->Shards, resp->Get()->Configs.Shards);
    }
}

TEST_F(TConfigsPullerTest, DbRespondsWithAnError) {
    {
        auto dao = MakeIntrusive<TShardDao>(EErrorPolicy::WithoutErrors);
        auto cp = RegisterConfigsPuller(dao);
        ActorRuntime->WaitForBootstrap();

        ActorRuntime->Send(cp, EdgeId, MakeHolder<TConfigsPullerEvents::TSubscribe>());
        auto resp = ActorRuntime->GrabEdgeEvent<TConfigsPullerEvents::TConfigsResponse>(EdgeId);
        ASSERT_TRUE(resp);
        EXPECT_EQ(dao->GetAllCnt, 1ul); // we got a successful response right after requesting the dao

        ActorRuntime->Send(cp, MakeHolder<TEvents::TEvPoison>());
    }

    {
        auto dao = MakeIntrusive<TShardDao>(EErrorPolicy::ErrorOnFirstCall);
        auto cp = RegisterConfigsPuller(dao);
        ActorRuntime->WaitForBootstrap();

        ActorRuntime->Send(cp, EdgeId, MakeHolder<TConfigsPullerEvents::TSubscribe>());
        auto resp = ActorRuntime->GrabEdgeEvent<TConfigsPullerEvents::TConfigsResponse>(EdgeId);
        ASSERT_TRUE(resp);
        EXPECT_EQ(dao->GetAllCnt, 2ul); // we got a successful response only after two attempts

        ActorRuntime->Send(cp, MakeHolder<TEvents::TEvPoison>());
    }
}

TEST_F(TConfigsPullerTest, MultipleSubscribers) {
    auto dao = MakeIntrusive<TShardDao>(EErrorPolicy::WithoutErrors);
    auto cp = RegisterConfigsPuller(dao);
    ActorRuntime->WaitForBootstrap();

    ActorRuntime->Send(cp, EdgeId, MakeHolder<TConfigsPullerEvents::TSubscribe>());

    ActorRuntime->AdvanceCurrentTime(TDuration::Seconds(11));

    size_t callCnt;
    {
        auto resp = ActorRuntime->GrabEdgeEvent<TConfigsPullerEvents::TConfigsResponse>(EdgeId, TDuration::MicroSeconds(1));
        ASSERT_TRUE(resp);

        EXPECT_GE(dao->GetAllCnt, 1ul);
        callCnt = dao->GetAllCnt;
        EXPECT_EQ(dao->Shards.size(), resp->Get()->Configs.Shards.size());
        EXPECT_EQ(dao->Shards, resp->Get()->Configs.Shards);
    }

    auto edge2 = ActorRuntime->AllocateEdgeActor();
    ActorRuntime->Send(cp, edge2, MakeHolder<TConfigsPullerEvents::TSubscribe>());

    { // new subscriber, but configs are from the cache
        auto resp = ActorRuntime->GrabEdgeEvent<TConfigsPullerEvents::TConfigsResponse>(edge2, TDuration::MicroSeconds(1));
        ASSERT_TRUE(resp);

        EXPECT_EQ(dao->GetAllCnt, callCnt);
        EXPECT_EQ(dao->Shards.size(), resp->Get()->Configs.Shards.size());
        EXPECT_EQ(dao->Shards, resp->Get()->Configs.Shards);
    }
}

TEST_F(TConfigsPullerTest, ConfigsForTesting) {
    TVector<NDb::NModel::TShardConfig> shards{
        {
            .Id = "shard1",
            .NumId = 1,
        },
        {
            .Id = "shard2",
            .NumId = 2,
        },
    };

    auto cp = RegisterConfigsPuller(nullptr, shards);
    ActorRuntime->WaitForBootstrap();
    ActorRuntime->Send(cp, EdgeId, MakeHolder<TConfigsPullerEvents::TSubscribe>());

    {
        auto resp = ActorRuntime->GrabEdgeEvent<TConfigsPullerEvents::TConfigsResponse>(EdgeId, TDuration::MicroSeconds(1));
        ASSERT_TRUE(resp);

        EXPECT_FALSE(resp->Get()->Configs.Shards.empty());
        EXPECT_EQ(resp->Get()->Configs.Shards, shards);
    }
}
