#include "table_loader.h"

#include <solomon/libs/cpp/logging/logging.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/events.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/core/log.h>
#include <library/cpp/monlib/metrics/labels.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/threading/future/future.h>

#include <util/generic/algorithm.h>
#include <util/system/hp_timer.h>

#include <utility>

namespace NSolomon::NTableLoader {
namespace {

using namespace NActors;
using namespace NDb::NModel;
using namespace NDb;
using namespace NMonitoring;
using namespace NThreading;


struct TDbCounters {
    TDbCounters(IMetricRegistry& registry, const TString& tablePath) {
        TLabel tableLabel{"table", tablePath};

        LoadTime = registry.IntGauge(MakeLabels({tableLabel, {"sensor", "loadTimeMillis"}}));
        Success = registry.Counter(MakeLabels({tableLabel, {"sensor", "success"}}));
        Fail = registry.Counter(MakeLabels({tableLabel, {"sensor", "fail"}}));
    }

    TDbCounters& operator=(const TDbCounters&) = default;
    TDbCounters(const TDbCounters&) = default;

    NMonitoring::IIntGauge* LoadTime{nullptr};
    NMonitoring::ICounter* Success{nullptr};
    NMonitoring::ICounter* Fail{nullptr};
};

template <typename TDerived, typename TTableLoaderTraits>
class TTableLoader: public TActorBootstrapped<TDerived>
{
    using TDaoPtr = typename TTableLoaderTraits::TDaoPtr;
    using TModel = typename TTableLoaderTraits::TModel;
    using TEvReady = typename TTableLoaderTraits::TEvReady;

public:
    // NOLINTNEXTLINE(performance-unnecessary-value-param): false positive
    TTableLoader(TDaoPtr dao, IMetricRegistry& registry, TString tableName, TActorId receiver)
        : Dao_{dao}
        , Counters_{registry, std::move(tableName)}
        , Receiver_{receiver}
    {
    }

    ~TTableLoader() {
        if (FutureResult_.Initialized()) {
            FutureResult_.Wait();
        }
    }

    void Bootstrap(const TActorContext& ctx) noexcept {
        TDerived::Become(&TDerived::StateFunc);

        const auto receiver = Receiver_;
        THPTimer timer;

        auto* failCount = Counters_.Fail;
        auto* successCount = Counters_.Success;
        auto* loadTime = Counters_.LoadTime;

        auto* as = ctx.ExecutorThread.ActorSystem;

        FutureResult_ = Dao_->GetAll().Apply([=, self = Self()->SelfId()] (auto f) {
            loadTime->Set(TDuration::FromValue(timer.Passed()).MilliSeconds());

            try {
                auto models = f.ExtractValue();
                successCount->Inc();
                as->Send(receiver, new TEvReady{std::move(models)});
            } catch (...) {
                failCount->Inc();
                MON_ERROR_C(*as, TableLoader, CurrentExceptionMessage());
                as->Send(receiver, new TEvReady{CurrentExceptionMessage()});
            }

            as->Send(self, new TEvents::TEvPoisonPill);
        });
    }

    STFUNC(StateFunc) {
        Y_UNUSED(ctx);

        switch (ev->GetTypeRewrite()) {
            cFunc(TEvents::TSystem::PoisonPill, TDerived::PassAway);
        };
    }
protected:
    TDaoPtr Dao_;
    TDbCounters Counters_;
    TFuture<void> FutureResult_;
    TActorId Receiver_;

private:
    TDerived* Self() {
        return static_cast<TDerived*>(this);
    }
};

struct TShardLoaderTraits {
    using TModel = NModel::TShardConfig;
    using TDaoPtr = IShardConfigDaoPtr;
    using TEvReady = TEvShardsReady;
};

class TShardLoaderActor:
    public TTableLoader<TShardLoaderActor, TShardLoaderTraits>
{
    using TBase = TTableLoader<TShardLoaderActor, TShardLoaderTraits>;

public:
    TShardLoaderActor(NDb::IShardConfigDaoPtr shardDao, TShardLoaderConfig config, TActorId receiver)
        : TBase{std::move(shardDao), config.Metrics, "Shards", receiver}
        , Config_{config}
    {
    }

private:
    TShardLoaderConfig Config_;
};

struct TClusterLoaderTraits {
    using TModel = NModel::TClusterConfig;
    using TDaoPtr = IClusterConfigDaoPtr;
    using TEvReady = TEvClustersReady;
};

class TClusterLoaderActor:
    public TTableLoader<TClusterLoaderActor, TClusterLoaderTraits>
{
    using TBase = TTableLoader<TClusterLoaderActor, TClusterLoaderTraits>;

public:
    TClusterLoaderActor(NDb::IClusterConfigDaoPtr shardDao, TClusterLoaderConfig config, TActorId receiver)
        : TBase{std::move(shardDao), config.Metrics, "Clusters", receiver}
        , Config_{config}
    {
    }

private:
    TClusterLoaderConfig Config_;
};

struct TServiceLoaderTraits {
    using TModel = NModel::TServiceConfig;
    using TDaoPtr = IServiceConfigDaoPtr;
    using TEvReady = TEvServicesReady;
};

class TServiceLoaderActor:
    public TTableLoader<TServiceLoaderActor, TServiceLoaderTraits>
{
    using TBase = TTableLoader<TServiceLoaderActor, TServiceLoaderTraits>;

public:
    TServiceLoaderActor(NDb::IServiceConfigDaoPtr shardDao, TServiceLoaderConfig config, TActorId receiver)
        : TBase{std::move(shardDao), config.Metrics, "Services", receiver}
        , Config_{config}
    {
    }

private:
    TServiceLoaderConfig Config_;
};

struct TProjectLoaderTraits {
    using TModel = NModel::TProjectConfig;
    using TDaoPtr = IProjectConfigDaoPtr;
    using TEvReady = TEvProjectsReady;
};

class TProjectLoaderActor:
    public TTableLoader<TProjectLoaderActor, TProjectLoaderTraits>
{
    using TBase = TTableLoader<TProjectLoaderActor, TProjectLoaderTraits>;

public:
    TProjectLoaderActor(NDb::IProjectConfigDaoPtr shardDao, TProjectLoaderConfig config, TActorId receiver)
        : TBase{std::move(shardDao), config.Metrics, "Projects", receiver}
        , Config_{config}
    {
    }

private:
    TProjectLoaderConfig Config_;
};

struct TAgentLoaderTraits {
    using TModel = NModel::TAgentConfig;
    using TDaoPtr = IAgentConfigDaoPtr;
    using TEvReady = TEvAgentsReady;
};

class TAgentLoaderActor:
    public TTableLoader<TAgentLoaderActor, TAgentLoaderTraits>
{
    using TBase = TTableLoader<TAgentLoaderActor, TAgentLoaderTraits>;

public:
    TAgentLoaderActor(NDb::IAgentConfigDaoPtr agentDao, TAgentLoaderConfig config, TActorId receiver)
        : TBase{std::move(agentDao), config.Metrics, "Agents", receiver}
        , Config_{config}
    {
    }

private:
    TAgentLoaderConfig Config_;
};

struct TProviderLoaderTraits {
    using TModel = NModel::TProviderConfig;
    using TDaoPtr = IProviderConfigDaoPtr;
    using TEvReady = TEvProvidersReady;
};

class TProviderLoaderActor
        : public TTableLoader<TProviderLoaderActor, TProviderLoaderTraits>
{
    using TBase = TTableLoader<TProviderLoaderActor, TProviderLoaderTraits>;

public:
    TProviderLoaderActor(NDb::IProviderConfigDaoPtr providerDao, TProviderLoaderConfig config, TActorId receiver)
        : TBase{std::move(providerDao), config.Metrics, "Providers", receiver}
        , Config_{config}
    {
    }

private:
    TProviderLoaderConfig Config_;
};

} // namespace

IActor* CreateShardLoaderActor(NDb::IShardConfigDaoPtr shardDao, TShardLoaderConfig config, TActorId receiver) {
    return new TShardLoaderActor{std::move(shardDao), config, receiver};
}

IActor* CreateClusterLoaderActor(NDb::IClusterConfigDaoPtr clusterDao, TClusterLoaderConfig config, TActorId receiver) {
    return new TClusterLoaderActor{std::move(clusterDao), config, receiver};
}

IActor* CreateServiceLoaderActor(NDb::IServiceConfigDaoPtr serviceDao, TServiceLoaderConfig config, TActorId receiver) {
    return new TServiceLoaderActor{std::move(serviceDao), config, receiver};
}

IActor* CreateProjectLoaderActor(NDb::IProjectConfigDaoPtr projectDao, TProjectLoaderConfig config, TActorId receiver) {
    return new TProjectLoaderActor{std::move(projectDao), config, receiver};
}

IActor* CreateAgentsLoaderActor(NDb::IAgentConfigDaoPtr agentDao, TAgentLoaderConfig config, TActorId receiver) {
    return new TAgentLoaderActor{std::move(agentDao), std::move(config), receiver};
}

IActor* CreateProvidersLoaderActor(NDb::IProviderConfigDaoPtr providerDao, TProviderLoaderConfig config, TActorId receiver) {
    return new TProviderLoaderActor{std::move(providerDao), std::move(config), receiver};
}

} // namespace NSolomon::NTableLoader
