#include "dao.h"

#include <solomon/libs/cpp/conf_db/model/agent_config.h>

#include <solomon/libs/cpp/ydb/dao_base.h>
#include <solomon/libs/cpp/ydb/paged_reader.h>
#include <solomon/libs/cpp/ydb/parser.h>
#include <solomon/libs/cpp/ydb/util.h>

#include <library/cpp/resource/resource.h>
#include <library/cpp/threading/future/future.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <utility>

using namespace NThreading;
using namespace NMonitoring;
using namespace NYdb;
using namespace NYdb::NTable;

namespace NSolomon::NDb {
namespace {
    constexpr TStringBuf CREATE_AGENT_TABLE = "agent_create_table.yql";
    constexpr TStringBuf DELETE_AGENT = "agent_delete.yql";
    constexpr TStringBuf DELETE_AGENTS_BY_PROVIDER = "agent_delete_for_provider.yql";
    constexpr TStringBuf DELETE_OBSOLETE_AGENTS = "agent_delete_obsolete.yql";
    constexpr TStringBuf INSERT_AGENT = "agent_insert.yql";

    class TAgentParser: public TModelParser<NModel::TAgentConfig> {
    public:
        TAgentParser(const TResultSet& rs)
            : TModelParser<NModel::TAgentConfig>{rs}
        {
            using namespace NModel;

            FieldMappings_["provider"] = Utf8FieldParser(Parser_, &TAgentConfig::Provider, "provider");
            FieldMappings_["hostname"] = Utf8FieldParser(Parser_, &TAgentConfig::Hostname, "hostname");
            FieldMappings_["dataPort"] = I32ToUI16FieldParser(Parser_, &TAgentConfig::DataPort, "dataPort");
            FieldMappings_["managementPort"] = I32ToUI16FieldParser(Parser_, &TAgentConfig::ManagementPort, "managementPort");
            FieldMappings_["version"] = Utf8FieldParser(Parser_, &TAgentConfig::Version, "version");
            FieldMappings_["lastSeen"] = I64ToInstantFieldParser(Parser_, &TAgentConfig::LastSeen, "lastSeen");
            FieldMappings_["pullIntervalSeconds"] = I32ToDurationFieldParser(Parser_, &TAgentConfig::PullInterval, "pullIntervalSeconds");
        }
    };

    auto UnwrapReadTableResult = ParseReadTableResult<NModel::TAgentConfig, TAgentParser>;

    class TAgentDao: public IAgentConfigDao, private TDaoBase {
    public:
        TAgentDao(TString path, std::shared_ptr<TTableClient> client, TMetricRegistry& registry)
            : TDaoBase{std::move(client), registry}
            , TablePath_{std::move(path)}
        {
            Init();
        }

        const TString& Name() const override {
            static const TString NAME{"AgentDao"};
            return NAME;
        }

        const TVector<TStringBuf>& Methods() const override {
            static const TVector<TStringBuf> METHODS{
                TStringBuf("createSchema"),
                TStringBuf("dropSchema"),
                TStringBuf("delete"),
                TStringBuf("deleteByProvider"),
                TStringBuf("deleteObsolete"),
                TStringBuf("findByProvider"),
                TStringBuf("insertOrUpdate"),
                TStringBuf("findAll"),
            };

            return METHODS;
        }

        TAsyncVoid CreateTable() override {
            return Execute<TAsyncStatus>(*Client_, [q = RawQueries_.at(CREATE_AGENT_TABLE)] (TSession session) {
                return session.ExecuteSchemeQuery(q);
            }).Apply(CompleteStatus(CounterFor("createSchema")));
        }

        TAsyncVoid DropTable() override {
            return Execute<TAsyncStatus>(*Client_, [=] (TSession session) {
                return session.DropTable(TablePath_);
            }).Apply(CompleteStatus(CounterFor("dropSchema")));
        }

        TAsyncVoid Delete(const TString& hostname, const TString& provider) override {
            const auto req = RawQueries_.at(DELETE_AGENT);
            auto params = TParamsBuilder{}
                    .AddParam("$hostname").Utf8(hostname).Build()
                    .AddParam("$provider").Utf8(provider).Build()
                .Build();

            return Client_->RetryOperation<TDataQueryResult>([=, params{std::move(params)}] (NYdb::NTable::TSession s) {
                return ExecutePrepared(std::move(s), req, params);
            }).Apply(CompleteStatus(CounterFor("delete")));
        }

        TAsyncVoid InsertOrUpdate(const NModel::TAgentConfig& model) override {
            auto q = RawQueries_.at(INSERT_AGENT);
            auto params = TParamsBuilder{}
                    .AddParam("$provider").Utf8(model.Provider).Build()
                    .AddParam("$hostname").Utf8(model.Hostname).Build()
                    .AddParam("$dataPort").Int32(model.DataPort).Build()
                    .AddParam("$managementPort").Int32(model.ManagementPort).Build()
                    .AddParam("$version").Utf8(model.Version).Build()
                    .AddParam("$lastSeen").Int64(model.LastSeen.MilliSeconds()).Build()
                    .AddParam("$pullIntervalSeconds").Int32(model.PullInterval.Seconds()).Build()
                    .AddParam("$description").Utf8(model.Description).Build()
            .Build();

            return Client_->RetryOperation<TDataQueryResult>([=, params{std::move(params)}] (NYdb::NTable::TSession s) {
                return ExecutePrepared(std::move(s), q, params);
            }).Apply(CompleteStatus(CounterFor("insertOrUpdate")));
        }

        TAsyncVoid DeleteByProvider(const TString& provider) override {
            auto q = RawQueries_.at(DELETE_AGENTS_BY_PROVIDER);
            auto params = TParamsBuilder{}
                    .AddParam("$provider").Utf8(provider).Build()
                .Build();

            return Client_->RetryOperation<TDataQueryResult>([=, params{std::move(params)}] (NYdb::NTable::TSession s) {
                return ExecutePrepared(std::move(s), q, params);
            }).Apply(CompleteStatus(CounterFor("deleteByProvider")));
        }

        TAsyncVoid DeleteObsolete(TInstant deadline) override {
            auto q = RawQueries_.at(DELETE_OBSOLETE_AGENTS);
            auto params = TParamsBuilder{}
                .AddParam("$deadline").Int64(deadline.Seconds()).Build()
            .Build();

            return Client_->RetryOperation<TDataQueryResult>([=, params{std::move(params)}] (NYdb::NTable::TSession s) {
                return ExecutePrepared(std::move(s), q, params);
            }).Apply(CompleteStatus(CounterFor("deleteObsolete")));
        }

        TAsyncAgentConfigs GetAll() override {
            return ReadTable<NModel::TAgentConfig>(*Client_, TablePath_, UnwrapReadTableResult, CounterFor("findAll"));
        }

    private:
        THashMap<TStringBuf, TString> LoadQueries() override {
            THashMap<TStringBuf, TString> result;

            auto getOne = [&] (TStringBuf key, TStringBuf resourceKey) {
                auto resource = NResource::Find(resourceKey);
                Y_ENSURE(!resource.empty(), "Resource " << resourceKey << " is empty");

                SubstGlobal(resource, TStringBuf("${agent.table.path}"), TablePath_);
                result.emplace(key, resource);
            };

            getOne(CREATE_AGENT_TABLE, "agent_create_table.yql");
            getOne(DELETE_AGENT, "agent_delete.yql");
            getOne(DELETE_AGENTS_BY_PROVIDER, "agent_delete_for_provider.yql");
            getOne(DELETE_OBSOLETE_AGENTS, "agent_delete_obsolete.yql");
            getOne(INSERT_AGENT, "agent_insert.yql");

            return result;
        }

    private:
        const TString TablePath_;
    };
} // namespace
    IAgentConfigDaoPtr CreateYdbAgentDao(const TString& path, std::shared_ptr<TTableClient> client, TMetricRegistry& registry) {
        return ::MakeIntrusive<TAgentDao>(path, std::move(client), registry);
    }
} // namespace NSolomon::NDb
