#include "dao.h"
#include "shard_settings_field_parser.h"

#include <solomon/libs/cpp/conf_db/model/cluster_config.h>
#include <solomon/libs/cpp/ydb/dao_base.h>
#include <solomon/libs/cpp/ydb/paged_reader.h>
#include <solomon/libs/cpp/ydb/parser.h>
#include <solomon/libs/cpp/ydb/util.h>

#include <library/cpp/resource/resource.h>
#include <library/cpp/threading/future/future.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/string/subst.h>
#include <util/generic/ptr.h>

#include <utility>

using namespace NThreading;
using namespace NMonitoring;
using namespace NYdb;
using namespace NYdb::NTable;


namespace NSolomon::NDb {
namespace {


const TString CREATE_CLUSTER_TABLE = "create_clusters";
const TString GET_CLUSTER_BY_PROJECT = "get_cluster_by_project";
const TString CLUSTER_FIND_BY_NAME = "cluster_find_by_name";
const TString INSERT_CLUSTER = "insert_cluster";
const TString DELETE_CLUSTER = "delete_cluster";


class TClusterParser: public TModelParser<NModel::TClusterConfig> {
public:
    explicit TClusterParser(const TResultSet& rs)
        : TModelParser<NModel::TClusterConfig>{rs}
    {
        using namespace NModel;

        FieldMappings_["id"] = Utf8FieldParser(Parser_, &TClusterConfig::Id, "id");
        FieldMappings_["name"] = Utf8FieldParser(Parser_, &TClusterConfig::Name, "name");
        FieldMappings_["description"] = Utf8FieldParser(Parser_, &TClusterConfig::Description, "description");
        FieldMappings_["projectId"] = Utf8FieldParser(Parser_, &TClusterConfig::ProjectId, "projectId");
        FieldMappings_["folderId"] = Utf8FieldParser(Parser_, &TClusterConfig::FolderId, "folderId");
        FieldMappings_["hosts"] = Utf8FieldParser(Parser_, &TClusterConfig::Hosts, "hosts");
        FieldMappings_["hostUrls"] = Utf8FieldParser(Parser_, &TClusterConfig::HostUrls, "hostUrls");
        FieldMappings_["conductorGroups"] = Utf8FieldParser(Parser_, &TClusterConfig::ConductorGroups, "conductorGroups");
        FieldMappings_["conductorTags"] = Utf8FieldParser(Parser_, &TClusterConfig::ConductorTags, "conductorTags");
        FieldMappings_["nannyGroups"] = Utf8FieldParser(Parser_, &TClusterConfig::NannyGroups, "nannyGroups");
        FieldMappings_["qloudGroups"] = Utf8FieldParser(Parser_, &TClusterConfig::QloudGroups, "qloudGroups");
        FieldMappings_["networks"] = Utf8FieldParser(Parser_, &TClusterConfig::Networks, "networks");
        FieldMappings_["ypClusters"] = Utf8FieldParser(Parser_, &TClusterConfig::YpClusters, "ypClusters");
        FieldMappings_["instanceGroups"] = Utf8FieldParser(Parser_, &TClusterConfig::InstanceGroups, "instanceGroups");
        FieldMappings_["cloudDns"] = Utf8FieldParser(Parser_, &TClusterConfig::CloudDns, "cloudDns");
        FieldMappings_["shardSettings"] = ShardSettingsFieldParser(Parser_, &TClusterConfig::ShardSettings, "shardSettings");
        FieldMappings_["labels"] = Utf8FieldParser(Parser_, &TClusterConfig::Labels, "labels");
        FieldMappings_["createdAt"] = I64ToInstantFieldParser(Parser_, &TClusterConfig::CreatedAt, "createdAt");
        FieldMappings_["updatedAt"] = I64ToInstantFieldParser(Parser_, &TClusterConfig::UpdatedAt, "updatedAt");
        FieldMappings_["createdBy"] = Utf8FieldParser(Parser_, &TClusterConfig::CreatedBy, "createdBy");
        FieldMappings_["updatedBy"] = Utf8FieldParser(Parser_, &TClusterConfig::UpdatedBy, "updatedBy");
        FieldMappings_["version"] = IToUI32FieldParser(Parser_, &TClusterConfig::Version, "version");
    }
};

auto UnwrapOne = ParseOne<NModel::TClusterConfig, TClusterParser>;
auto UnwrapReadTableResult = ParseReadTableResult<NModel::TClusterConfig, TClusterParser>;

class TClusterDao: public IClusterConfigDao, private TDaoBase {
public:
    TClusterDao(TString path, std::shared_ptr<TTableClient> client, TMetricRegistry& registry)
        : TDaoBase{std::move(client), registry}
        , TablePath_{std::move(path)}
    {
        Init();
    }

    const TString& Name() const override {
        static const TString NAME{"ClustersDao"};
        return NAME;
    }

    const TVector<TStringBuf>& Methods() const override {
        static const TVector<TStringBuf> METHODS{
            TStringBuf("createSchema"),
            TStringBuf("dropSchema"),
            TStringBuf("deleteByProject"),
            TStringBuf("findByProject"),
            TStringBuf("findByName"),
            TStringBuf("insert"),
            TStringBuf("findAll"),
        };

        return METHODS;
    }

    TAsyncVoid CreateTable() override {
        return Execute<TAsyncStatus>(*Client_, [q = RawQueries_.at(CREATE_CLUSTER_TABLE)] (TSession session) {
            return session.ExecuteSchemeQuery(q);
        }).Apply(CompleteStatus(CounterFor("createSchema")));
    }

    TAsyncVoid DropTable() override {
        return Execute<TAsyncStatus>(*Client_, [=] (TSession session) {
            return session.DropTable(TablePath_);
        }).Apply(CompleteStatus(CounterFor("dropSchema")));
    }

    TAsyncVoid Delete(const TString& clusterId, const TString& projectId) override {
        const auto req = RawQueries_.at(DELETE_CLUSTER);
        auto params = TParamsBuilder{}
                .AddParam("$id").Utf8(clusterId).Build()
                .AddParam("$projectId").Utf8(projectId).Build()
                .AddParam("$folderId").Utf8("").Build()
            .Build();

        return Client_->RetryOperation<TDataQueryResult>([=, params{std::move(params)}] (NYdb::NTable::TSession s) {
            return ExecutePrepared(std::move(s), req, params);
        }).Apply(CompleteStatus(CounterFor("deleteByProject")));
    }

    TAsyncVoid Insert(const NModel::TClusterConfig& model) override {
        auto q = RawQueries_.at(INSERT_CLUSTER);
        auto params = TParamsBuilder{}
                .AddParam("$id").Utf8(model.Id).Build()
                .AddParam("$name").Utf8(model.Name).Build()
                .AddParam("$description").Utf8(model.Description).Build()
                .AddParam("$projectId").Utf8(model.ProjectId).Build()
                .AddParam("$folderId").Utf8("").Build()
                .AddParam("$hosts").Utf8(model.Hosts).Build()
                .AddParam("$hostUrls").Utf8(model.HostUrls).Build()
                .AddParam("$conductorGroups").Utf8(model.ConductorGroups).Build()
                .AddParam("$conductorTags").Utf8(model.ConductorTags).Build()
                .AddParam("$nannyGroups").Utf8(model.NannyGroups).Build()
                .AddParam("$qloudGroups").Utf8(model.QloudGroups).Build()
                .AddParam("$networks").Utf8(model.Networks).Build()
                .AddParam("$ypClusters").Utf8(model.YpClusters).Build()
                .AddParam("$instanceGroups").Utf8(model.InstanceGroups).Build()
                .AddParam("$cloudDns").Utf8(model.CloudDns).Build()
                .AddParam("$shardSettings").Utf8(model.ShardSettings.ToJsonStr()).Build()
                .AddParam("$labels").Utf8(model.Labels).Build()
                .AddParam("$createdAt").Int64(model.CreatedAt.MilliSeconds()).Build()
                .AddParam("$updatedAt").Int64(model.UpdatedAt.MilliSeconds()).Build()
                .AddParam("$createdBy").Utf8(model.CreatedBy).Build()
                .AddParam("$updatedBy").Utf8(model.UpdatedBy).Build()
                .AddParam("$version").Int32(model.Version).Build()
                // TODO: remove these stubs
                .AddParam("$port").Int32(0).Build()
                .AddParam("$useFqdn").Bool(false).Build()
                .AddParam("$tvmDestId").Utf8(TString{}).Build()
                .AddParam("$sensorsTtlDays").Int32(0).Build()
            .Build();

        return Client_->RetryOperation<TDataQueryResult>([=, params{std::move(params)}] (NYdb::NTable::TSession s) {
            return ExecutePrepared(std::move(s), q, params);
        }).Apply(CompleteStatus(CounterFor("insert")));
    }

    TAsyncClusterConfig GetByProject(const TString& clusterId, const TString& projectId) override {
        auto q = RawQueries_.at(GET_CLUSTER_BY_PROJECT);
        auto params = TParamsBuilder{}
                .AddParam("$projectId").Utf8(projectId).Build()
                .AddParam("$folderId").Utf8("").Build()
                .AddParam("$id").Utf8(clusterId).Build()
            .Build();

        auto p = NewPromise<std::optional<NModel::TClusterConfig>>();
        ReadDataWithRetries(*Client_, q, std::move(params), p, UnwrapOne, CounterFor("findByProject"));
        return p.GetFuture();
    }

    TAsyncClusterConfig GetByName(const TString& clusterName, const TString& projectId) override {
        auto q = RawQueries_.at(CLUSTER_FIND_BY_NAME);
        auto params = TParamsBuilder{}
                .AddParam("$projectId").Utf8(projectId).Build()
                .AddParam("$folderId").Utf8("").Build()
                .AddParam("$name").Utf8(clusterName).Build()
            .Build();

        auto p = NewPromise<std::optional<NModel::TClusterConfig>>();
        ReadDataWithRetries(*Client_, q, std::move(params), p, UnwrapOne, CounterFor("findByName"));
        return p.GetFuture();
    }

    TAsyncClusterConfigs GetAll() override {
        return ReadTable<NModel::TClusterConfig>(*Client_, TablePath_, UnwrapReadTableResult, CounterFor("findAll"));
    }

private:
    THashMap<TStringBuf, TString> LoadQueries() override {
        THashMap<TStringBuf, TString> result;

        auto getOne = [&] (TStringBuf key, TStringBuf resourceKey) {
            auto resource = NResource::Find(resourceKey);
            Y_ENSURE(!resource.empty(), "Resource " << resourceKey << " is empty");

            SubstGlobal(resource, TStringBuf("${cluster.table.path}"), TablePath_);
            result.emplace(key, resource);
        };

        getOne(CREATE_CLUSTER_TABLE, "cluster_create_table.yql");
        getOne(GET_CLUSTER_BY_PROJECT, "cluster_find.yql");
        getOne(CLUSTER_FIND_BY_NAME, "cluster_find_by_name.yql");
        getOne(DELETE_CLUSTER, "cluster_delete.yql");
        getOne(INSERT_CLUSTER, "cluster_insert.yql");

        return result;
    }

private:
    const TString TablePath_;
};

} // namespace

IClusterConfigDaoPtr CreateYdbClusterDao(const TString& path, std::shared_ptr<TTableClient> client, TMetricRegistry& registry) {
    return ::MakeIntrusive<TClusterDao>(path, std::move(client), registry);
}

} // namespace NSolomon::NDb
