#include "dao.h"

#include <solomon/libs/cpp/conf_db/model/project_config.h>
#include <solomon/libs/cpp/ydb/dao_base.h>
#include <solomon/libs/cpp/ydb/paged_reader.h>
#include <solomon/libs/cpp/ydb/parser.h>
#include <solomon/libs/cpp/ydb/util.h>

#include <library/cpp/resource/resource.h>
#include <library/cpp/threading/future/future.h>

#include <util/generic/ptr.h>
#include <util/string/builder.h>
#include <util/string/vector.h>
#include <util/string/subst.h>

#include <utility>

using namespace NMonitoring;
using namespace NThreading;
using namespace NYdb::NTable;
using namespace NYdb;

namespace NSolomon::NDb {
namespace {

const TString PROJECT_CREATE_TABLE = "project_create_table";
const TString PROJECT_DELETE = "project_delete";
const TString PROJECT_INSERT = "project_insert";
const TString PROJECT_EXISTS = "project_exists";
const TString PROJECT_FIND_BY_ID = "project_find";


class TProjectParser: public TModelParser<NModel::TProjectConfig> {
public:
    explicit TProjectParser(const TResultSet& rs)
        : TModelParser<NModel::TProjectConfig>{rs}
    {
        using namespace NModel;

        FieldMappings_["id"] = Utf8FieldParser(Parser_, &TProjectConfig::Id, "id");
        FieldMappings_["name"] = Utf8FieldParser(Parser_, &TProjectConfig::Name, "name");
        FieldMappings_["description"] = Utf8FieldParser(Parser_, &TProjectConfig::Description, "description");
        FieldMappings_["owner"] = Utf8FieldParser(Parser_, &TProjectConfig::Owner, "owner");
        FieldMappings_["abcService"] = Utf8FieldParser(Parser_, &TProjectConfig::AbcService, "abcService");
        FieldMappings_["deleteAcl"] = Utf8FieldParser(Parser_, &TProjectConfig::DeleteAcl, "deleteAcl");
        FieldMappings_["updateAcl"] = Utf8FieldParser(Parser_, &TProjectConfig::UpdateAcl, "updateAcl");
        FieldMappings_["readAcl"] = Utf8FieldParser(Parser_, &TProjectConfig::ReadAcl, "readAcl");
        FieldMappings_["writeAcl"] = Utf8FieldParser(Parser_, &TProjectConfig::WriteAcl, "writeAcl");
        FieldMappings_["createdAt"] = I64ToInstantFieldParser(Parser_, &TProjectConfig::CreatedAt, "createdAt");
        FieldMappings_["updatedAt"] = I64ToInstantFieldParser(Parser_, &TProjectConfig::UpdatedAt, "updatedAt");
        FieldMappings_["createdBy"] = Utf8FieldParser(Parser_, &TProjectConfig::CreatedBy, "createdBy");
        FieldMappings_["updatedBy"] = Utf8FieldParser(Parser_, &TProjectConfig::UpdatedBy, "updatedBy");
        FieldMappings_["version"] = IToUI32FieldParser(Parser_, &TProjectConfig::Version, "version");
        FieldMappings_["onlyAuthPush"] = BoolFieldParser(Parser_, &TProjectConfig::OnlyAuthPush, "onlyAuthPush");
        FieldMappings_["onlyAuthRead"] = BoolFieldParser(Parser_, &TProjectConfig::OnlyAuthRead, "onlyAuthRead");
        FieldMappings_["onlyNewFormatReads"] = BoolFieldParser(Parser_, &TProjectConfig::OnlyNewFormatReads, "onlyNewFormatReads");
        FieldMappings_["onlyNewFormatWrites"] = BoolFieldParser(Parser_, &TProjectConfig::OnlyNewFormatWrites, "onlyNewFormatWrites");
        FieldMappings_["onlySensorNameShards"] = BoolFieldParser(Parser_, &TProjectConfig::OnlyMetricNameShards, "onlySensorNameShards");
        FieldMappings_["metricNameLabel"] = Utf8FieldParser(Parser_, &TProjectConfig::MetricNameLabel, "metricNameLabel");
        FieldMappings_["labels"] = Utf8FieldParser(Parser_, &TProjectConfig::Labels, "labels");

    }
};

auto UnwrapReadTableResult = ParseReadTableResult<NModel::TProjectConfig, TProjectParser>;
auto UnwrapOne = ParseOne<NModel::TProjectConfig, TProjectParser>;

bool UnwrapExistsResult(const NYdb::NTable::TDataQueryResult& result) {
    auto&& resultSets = result.GetResultSets();

    Y_ENSURE(resultSets.size() <= 1);
    if (resultSets.size() == 0) {
        return {};
    }

    return resultSets[0].RowsCount() > 0;
}

class TProjectDao: public IProjectConfigDao, private TDaoBase {
public:
    TProjectDao(TString table, std::shared_ptr<TTableClient> client, TMetricRegistry& registry)
        : TDaoBase{std::move(client), registry}
        , TablePath_{std::move(table)}
    {
        Init();
    }

    const TString& Name() const override {
        static const TString NAME{"ProjectDao"};
        return NAME;
    }

    TAsyncVoid CreateTable() override {
        const auto req = RawQueries_.at(PROJECT_CREATE_TABLE);

        return Execute<TAsyncStatus>(*Client_, [req] (TSession session) {
            return session.ExecuteSchemeQuery(req);
        }).Apply(CompleteStatus(CounterFor("createSchema")));
    }

    TAsyncVoid DropTable() override {
        return Execute<TAsyncStatus>(*Client_, [=] (TSession session) {
            return session.DropTable(TablePath_);
        }).Apply(CompleteStatus(CounterFor("dropSchema")));
    }

    TAsyncVoid Insert(const NModel::TProjectConfig& model) override {
        auto req = RawQueries_.at(PROJECT_INSERT);

        auto params = TParamsBuilder{}
                .AddParam("$id").Utf8(model.Id).Build()
                .AddParam("$name").Utf8(model.Name).Build()
                .AddParam("$description").Utf8(model.Description).Build()
                .AddParam("$owner").Utf8(model.Owner).Build()
                .AddParam("$abcService").Utf8(model.AbcService).Build()
                .AddParam("$version").Int32(model.Version).Build()
                .AddParam("$createdAt").Int64(model.CreatedAt.MilliSeconds()).Build()
                .AddParam("$createdBy").Utf8(model.CreatedBy).Build()
                .AddParam("$updatedAt").Int64(model.UpdatedAt.MilliSeconds()).Build()
                .AddParam("$updatedBy").Utf8(model.UpdatedBy).Build()
                .AddParam("$deleteAcl").Utf8(model.DeleteAcl).Build()
                .AddParam("$updateAcl").Utf8(model.UpdateAcl).Build()
                .AddParam("$readAcl").Utf8(model.ReadAcl).Build()
                .AddParam("$writeAcl").Utf8(model.WriteAcl).Build()
                .AddParam("$onlyAuthPush").Bool(model.OnlyAuthPush).Build()
                .AddParam("$onlyAuthRead").Bool(model.OnlyAuthRead).Build()
                .AddParam("$onlySensorNameShards").Bool(model.OnlyMetricNameShards).Build()
                .AddParam("$onlyNewFormatWrites").Bool(model.OnlyNewFormatWrites).Build()
                .AddParam("$onlyNewFormatReads").Bool(model.OnlyNewFormatReads).Build()
                .AddParam("$metricNameLabel").Utf8(model.MetricNameLabel).Build()
                .AddParam("$labels").Utf8(model.Labels).Build()
            .Build();

        return Client_->RetryOperation<TDataQueryResult>([=] (NYdb::NTable::TSession s) {
            return ExecutePrepared(std::move(s), req, params);
        }).Apply(CompleteStatus(CounterFor("insert")));
    }

    TAsyncVoid Delete(const TString& projectId) override {
        const auto req = RawQueries_.at(PROJECT_DELETE);
        auto params = TParamsBuilder{}
                .AddParam("$id").Utf8(projectId).Build()
            .Build();

        return Client_->RetryOperation<TDataQueryResult>([=] (NYdb::NTable::TSession s) {
            return ExecutePrepared(std::move(s), req, params);
        }).Apply(CompleteStatus(CounterFor("delete")));
    }

    TAsyncProjectConfig GetById(const TString& projectId) override {
        const auto req = RawQueries_.at(PROJECT_FIND_BY_ID);
        auto params = TParamsBuilder{}
                .AddParam("$id").Utf8(projectId).Build()
            .Build();

        auto p = NewPromise<std::optional<NModel::TProjectConfig>>();
        ReadDataWithRetries(*Client_, req, std::move(params), p, UnwrapOne, CounterFor("findById"));
        return p;
    }

    TAsyncBool Exists(const TString& projectId) override {
        const auto req = RawQueries_.at(PROJECT_EXISTS);
        auto params = TParamsBuilder{}
                .AddParam("$id").Utf8(projectId).Build()
            .Build();

        auto p = NewPromise<bool>();
        ReadDataWithRetries(*Client_, req, std::move(params), p, UnwrapExistsResult, CounterFor("exists"));
        return p;
    }

    TAsyncProjectConfigs GetAll() override {
        return ReadTable<NModel::TProjectConfig>(*Client_, TablePath_, UnwrapReadTableResult, CounterFor("findAll"));
    }

    const TVector<TStringBuf>& Methods() const override {
        static const TVector<TStringBuf> METHODS{
            TStringBuf("createSchema"),
            TStringBuf("dropSchema"),
            TStringBuf("insert"),
            TStringBuf("delete"),
            TStringBuf("exists"),
            TStringBuf("findAll"),
            TStringBuf("findById"),
        };

        return METHODS;
    }

private:
    THashMap<TStringBuf, TString> LoadQueries() override {
        THashMap<TStringBuf, TString> result;

        auto getOne = [&] (TStringBuf key) {
            const auto resourceKey = TString{key} + ".yql";
            auto resource = NResource::Find(resourceKey);
            Y_ENSURE(!resource.empty(), "Resource " << resourceKey << " is empty");

            SubstGlobal(resource, TStringBuf("${project.table.path}"), TablePath_);
            result.emplace(key, resource);
        };

        getOne(PROJECT_CREATE_TABLE);
        getOne(PROJECT_DELETE);
        getOne(PROJECT_INSERT);
        getOne(PROJECT_EXISTS);
        getOne(PROJECT_FIND_BY_ID);

        return result;
    }

private:
    const TString TablePath_;
};

} // namespace

IProjectConfigDaoPtr CreateYdbProjectDao(TString table, std::shared_ptr<TTableClient> client, TMetricRegistry& registry) {
    return ::MakeIntrusive<TProjectDao>(std::move(table), std::move(client), registry);
}
} // namespace NSolomon::NDb
