#include "dao.h"

#include <solomon/libs/cpp/conf_db/model/provider_config.h>
#include <solomon/libs/cpp/ydb/dao_base.h>
#include <solomon/libs/cpp/ydb/paged_reader.h>
#include <solomon/libs/cpp/ydb/parser.h>
#include <solomon/libs/cpp/ydb/util.h>

#include <util/string/join.h>

#include <utility>

using namespace NMonitoring;
using namespace NThreading;
using namespace NYdb::NTable;
using namespace NYdb;

namespace NSolomon::NDb {
namespace {

constexpr TStringBuf SERVICE_PROVIDER_CREATE_TABLE = "service_provider_create_table";
constexpr TStringBuf SERVICE_PROVIDER_DELETE = "service_provider_delete";
constexpr TStringBuf SERVICE_PROVIDER_INSERT = "service_provider_insert";
constexpr TStringBuf SERVICE_PROVIDER_EXISTS = "service_provider_exists";
constexpr TStringBuf SERVICE_PROVIDER_FIND_BY_ID = "service_provider_find";

TVector<TString> TsvToList(TStringBuf tsv) {
    return StringSplitter(tsv).Split('\t').SkipEmpty();
}

TVector<NTvmAuth::TTvmId> TsvToTvmIdList(TStringBuf tsv) {
    TVector<NTvmAuth::TTvmId> result;

    for (TStringBuf src: StringSplitter(tsv).Split('\t').SkipEmpty()) {
        result.emplace_back(FromString<NTvmAuth::TTvmId>(src));
    }

    return result;
}

template <typename T>
TString ListToTsv(const TVector<T>& list) {
    return JoinSeq(TStringBuf("\t"), list);
}

class TProviderParser: public TModelParser<NModel::TProviderConfig> {
    template <typename TModel>
    auto Utf8ToTShardSettingsFieldParser(
            NYdb::TResultSetParser& parser,
            NModel::TProviderShardSettings (TModel::*field),
            const TString& column)
    {
        auto getter = [=, &parser] {
            return parser.ColumnParser(column).GetOptionalUtf8();
        };

        auto setter = [=] (TModel& model, const TString& value) {
            (model.*field) = NModel::TProviderShardSettings::FromString(value);
        };

        return MakeHolder<TIndirectFieldParserImpl<TModel, decltype(setter), decltype(getter)>>
                (std::move(getter), std::move(setter));
    }
    template <typename TModel>
    auto Utf8ToTReferenceConfFieldParser(
        NYdb::TResultSetParser& parser,
        TVector<NModel::TReferenceConf> (TModel::*field),
        const TString& column)
    {
        auto getter = [=, &parser] {
            return parser.ColumnParser(column).GetOptionalUtf8();
        };

        auto setter = [=] (TModel& model, const TString& value) {
            (model.*field) = NModel::TReferenceConf::FromStringArray(value);
        };

        return MakeHolder<TIndirectFieldParserImpl<TModel, decltype(setter), decltype(getter)>>
            (std::move(getter), std::move(setter));
    }

    template <typename TModel>
    auto Utf8ToStringListParser(
            NYdb::TResultSetParser& parser,
            TVector<TString> (TModel::*field),
            const TString& column)
    {
        auto getter = [=, &parser] {
            return parser.ColumnParser(column).GetOptionalUtf8();
        };

        auto setter = [=] (TModel& model, const TString& value) {
            (model.*field) = TsvToList(value);
        };

        return MakeHolder<TIndirectFieldParserImpl<TModel, decltype(setter), decltype(getter)>>
                (std::move(getter), std::move(setter));
    }

    template <typename TModel>
    auto Utf8ToTvmIdListParser(
            NYdb::TResultSetParser& parser,
            TVector<NTvmAuth::TTvmId> (TModel::*field),
            const TString& column)
    {
        auto getter = [=, &parser] {
            return parser.ColumnParser(column).GetOptionalUtf8();
        };

        auto setter = [=] (TModel& model, const TString& value) {
            (model.*field) = TsvToTvmIdList(value);
        };

        return MakeHolder<TIndirectFieldParserImpl<TModel, decltype(setter), decltype(getter)>>
                (std::move(getter), std::move(setter));
    }

public:
    explicit TProviderParser(const TResultSet& rs)
        : TModelParser<NModel::TProviderConfig>{rs}
    {
        using namespace NModel;

        FieldMappings_["id"] = Utf8FieldParser(Parser_, &TProviderConfig::Id, "id");
        FieldMappings_["description"] = Utf8FieldParser(Parser_, &TProviderConfig::Description, "description");
        FieldMappings_["shardSettings"] = Utf8ToTShardSettingsFieldParser(Parser_, &TProviderConfig::ShardSettings, "shardSettings");
        FieldMappings_["referenceSettings"] = Utf8ToTReferenceConfFieldParser(Parser_, &TProviderConfig::References, "referenceSettings");
        FieldMappings_["abcService"] = Utf8FieldParser(Parser_, &TProviderConfig::AbcService, "abcService");
        FieldMappings_["cloudId"] = Utf8FieldParser(Parser_, &TProviderConfig::CloudId, "cloudId");
        FieldMappings_["tvmDestId"] = Utf8FieldParser(Parser_, &TProviderConfig::TvmDestId, "tvmDestId");
        FieldMappings_["iamServiceAccountId"] = Utf8ToStringListParser(Parser_, &TProviderConfig::IamServiceAccountIds, "iamServiceAccountId");
        FieldMappings_["tvmServiceIds"] = Utf8ToTvmIdListParser(Parser_, &TProviderConfig::TvmServiceIds, "tvmServiceIds");
        FieldMappings_["createdAt"] = I64ToInstantFieldParser(Parser_, &TProviderConfig::CreatedAt, "createdAt");
        FieldMappings_["updatedAt"] = I64ToInstantFieldParser(Parser_, &TProviderConfig::UpdatedAt, "updatedAt");
        FieldMappings_["createdBy"] = Utf8FieldParser(Parser_, &TProviderConfig::CreatedBy, "createdBy");
        FieldMappings_["updatedBy"] = Utf8FieldParser(Parser_, &TProviderConfig::UpdatedBy, "updatedBy");
        FieldMappings_["version"] = IToUI32FieldParser(Parser_, &TProviderConfig::Version, "version");
    }
};

auto UnwrapOne = ParseOne<NModel::TProviderConfig, TProviderParser>;
auto UnwrapReadTableResult = ParseReadTableResult<NModel::TProviderConfig, TProviderParser>;

bool UnwrapExistsResult(const NYdb::NTable::TDataQueryResult& result) {
    auto&& resultSets = result.GetResultSets();

    Y_ENSURE(resultSets.size() <= 1);
    if (resultSets.size() == 0) {
        return {};
    }

    return resultSets[0].RowsCount() > 0;
}

class TProviderDao: public IProviderConfigDao, private TDaoBase {
public:
    TProviderDao(TString table, std::shared_ptr<TTableClient> client, TMetricRegistry& registry)
        : TDaoBase{std::move(client), registry}
        , TablePath_{std::move(table)}
    {
        Init();
    }

    const TString& Name() const override {
        static const TString NAME{"ProviderDao"};
        return NAME;
    }

    TAsyncVoid CreateTable() override {
        const auto req = RawQueries_.at(SERVICE_PROVIDER_CREATE_TABLE);

        return Execute<TAsyncStatus>(*Client_, [req] (TSession session) {
            return session.ExecuteSchemeQuery(req);
        }).Apply(CompleteStatus(CounterFor("createSchema")));
    }

    TAsyncVoid DropTable() override {
        return Execute<TAsyncStatus>(*Client_, [=] (TSession session) {
            return session.DropTable(TablePath_);
        }).Apply(CompleteStatus(CounterFor("dropSchema")));
    }

    TAsyncVoid Insert(const NModel::TProviderConfig& model) override {
        auto req = RawQueries_.at(SERVICE_PROVIDER_INSERT);

        auto params = TParamsBuilder{}
                .AddParam("$id").Utf8(model.Id).Build()
                .AddParam("$description").Utf8(model.Description).Build()
                .AddParam("$shardSettings").Utf8(model.ShardSettings.ToString()).Build()
                .AddParam("$referenceSettings").Utf8(NModel::TReferenceConf::ToStringArray(model.References)).Build()
                .AddParam("$abcService").Utf8(model.AbcService).Build()
                .AddParam("$cloudId").Utf8(model.CloudId).Build()
                .AddParam("$tvmDestId").Utf8(model.TvmDestId).Build()
                .AddParam("$iamServiceAccountId").Utf8(ListToTsv(model.IamServiceAccountIds)).Build()
                .AddParam("$tvmServiceIds").Utf8(ListToTsv(model.TvmServiceIds)).Build()
                .AddParam("$createdAt").Int64(model.CreatedAt.MilliSeconds()).Build()
                .AddParam("$updatedAt").Int64(model.UpdatedAt.MilliSeconds()).Build()
                .AddParam("$createdBy").Utf8(model.CreatedBy).Build()
                .AddParam("$updatedBy").Utf8(model.UpdatedBy).Build()
                .AddParam("$version").Int32(model.Version).Build()
                .Build();

        return Client_->RetryOperation<TDataQueryResult>([=] (NYdb::NTable::TSession s) {
            return ExecutePrepared(std::move(s), req, params);
        }).Apply(CompleteStatus(CounterFor("insert")));
    }

    TAsyncVoid Delete(const TString& providerId) override {
        const auto req = RawQueries_.at(SERVICE_PROVIDER_DELETE);
        auto params = TParamsBuilder{}
                .AddParam("$id").Utf8(providerId).Build()
                .Build();

        return Client_->RetryOperation<TDataQueryResult>([=] (NYdb::NTable::TSession s) {
            return ExecutePrepared(std::move(s), req, params);
        }).Apply(CompleteStatus(CounterFor("delete")));
    }

    TAsyncProviderConfig GetById(const TString& providerId) override {
        const auto req = RawQueries_.at(SERVICE_PROVIDER_FIND_BY_ID);
        auto params = TParamsBuilder{}
                .AddParam("$id").Utf8(providerId).Build()
                .Build();

        auto p = NewPromise<std::optional<NModel::TProviderConfig>>();
        ReadDataWithRetries(*Client_, req, std::move(params), p, UnwrapOne, CounterFor("findById"));
        return p;
    }

    TAsyncBool Exists(const TString& providerId) override {
        const auto req = RawQueries_.at(SERVICE_PROVIDER_EXISTS);
        auto params = TParamsBuilder{}
                .AddParam("$id").Utf8(providerId).Build()
                .Build();

        auto p = NewPromise<bool>();
        ReadDataWithRetries(*Client_, req, std::move(params), p, UnwrapExistsResult, CounterFor("exists"));
        return p;
    }

    TAsyncProviderConfigs GetAll() override {
        return ReadTable<NModel::TProviderConfig>(*Client_, TablePath_, UnwrapReadTableResult, CounterFor("findAll"));
    }

    const TVector<TStringBuf>& Methods() const override {
        static const TVector<TStringBuf> METHODS{
                TStringBuf("createSchema"),
                TStringBuf("dropSchema"),
                TStringBuf("insert"),
                TStringBuf("delete"),
                TStringBuf("exists"),
                TStringBuf("findAll"),
                TStringBuf("findById"),
        };

        return METHODS;
    }

private:
    THashMap<TStringBuf, TString> LoadQueries() override {
        THashMap<TStringBuf, TString> result;

        auto getOne = [&] (TStringBuf key) {
            const auto resourceKey = TString{key} + ".yql";
            auto resource = NResource::Find(resourceKey);
            Y_ENSURE(!resource.empty(), "Resource " << resourceKey << " is empty");

            SubstGlobal(resource, TStringBuf("${service.provider.table.path}"), TablePath_);
            result.emplace(key, resource);
        };

        getOne(SERVICE_PROVIDER_CREATE_TABLE);
        getOne(SERVICE_PROVIDER_DELETE);
        getOne(SERVICE_PROVIDER_INSERT);
        getOne(SERVICE_PROVIDER_EXISTS);
        getOne(SERVICE_PROVIDER_FIND_BY_ID);

        return result;
    }

private:
    const TString TablePath_;
};

} // namespace

IProviderConfigDaoPtr CreateYdbProviderDao(TString table, std::shared_ptr<TTableClient> client, TMetricRegistry& registry) {
    return ::MakeIntrusive<TProviderDao>(std::move(table), std::move(client), registry);
}

} // namespace NSolomon::NDb
