#include "dao.h"
#include "shard_settings_field_parser.h"

#include <solomon/libs/cpp/conf_db/model/service_config.h>
#include <solomon/libs/cpp/ydb/parser.h>
#include <solomon/libs/cpp/ydb/paged_reader.h>
#include <solomon/libs/cpp/ydb/util.h>
#include <solomon/libs/cpp/ydb/dao_base.h>

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/monlib/metrics/timer.h>

#include <library/cpp/resource/resource.h>
#include <library/cpp/threading/future/future.h>

#include <utility>

using namespace NThreading;
using namespace NMonitoring;
using namespace NYdb::NTable;
using namespace NYdb;

namespace NSolomon::NDb {
namespace {

const TString SERVICE_CREATE_TABLE = "service_create_table";
const TString SERVICE_FIND_WITH_PROJECT = "service_find_with_project";
const TString SERVICE_FIND_BY_NAME = "service_find_by_name";
const TString SERVICE_DELETE_FOR_PROJECT = "service_delete_for_project";
const TString SERVICE_DELETE_WITH_PROJECT = "service_delete_with_project";
const TString SERVICE_INSERT = "service_insert";


class TServiceParser: public TModelParser<NModel::TServiceConfig> {
    template <typename TModel>
    auto Utf8ToTMetricConfFieldParser(
        NYdb::TResultSetParser& parser,
        NModel::TMetricConf (TModel::*field), const TString& column)
    {
        auto getter = [=, &parser] {
            return parser.ColumnParser(column).GetOptionalUtf8();
        };

        auto setter = [=] (TModel& model, const TString& value) {
            (model.*field) = NModel::TMetricConf::FromString(value);
        };

        return MakeHolder<TIndirectFieldParserImpl<TModel, decltype(setter), decltype(getter)>>
            (std::move(getter), std::move(setter));
    }

public:
    explicit TServiceParser(const TResultSet& rs)
        : TModelParser<NModel::TServiceConfig>{rs}
    {
        using namespace NModel;

        FieldMappings_["id"] = Utf8FieldParser(Parser_, &TServiceConfig::Id, "id");
        FieldMappings_["name"] = Utf8FieldParser(Parser_, &TServiceConfig::Name, "name");
        FieldMappings_["description"] = Utf8FieldParser(Parser_, &TServiceConfig::Description, "description");
        FieldMappings_["projectId"] = Utf8FieldParser(Parser_, &TServiceConfig::ProjectId, "projectId");
        FieldMappings_["folderId"] = Utf8FieldParser(Parser_, &TServiceConfig::FolderId, "folderId");
        FieldMappings_["serviceProvider"] = Utf8FieldParser(Parser_, &TServiceConfig::ServiceProvider, "serviceProvider");
        FieldMappings_["createdAt"] = I64ToInstantFieldParser(Parser_, &TServiceConfig::CreatedAt, "createdAt");
        FieldMappings_["updatedAt"] = I64ToInstantFieldParser(Parser_, &TServiceConfig::UpdatedAt, "updatedAt");
        FieldMappings_["createdBy"] = Utf8FieldParser(Parser_, &TServiceConfig::CreatedBy, "createdBy");
        FieldMappings_["updatedBy"] = Utf8FieldParser(Parser_, &TServiceConfig::UpdatedBy, "updatedBy");
        FieldMappings_["version"] = IToUI32FieldParser(Parser_, &TServiceConfig::Version, "version");
        FieldMappings_["interval"] = I32ToDurationFieldParser(Parser_, &TServiceConfig::Interval, "interval");
        FieldMappings_["gridSec"] = Int32FieldParser(Parser_, &TServiceConfig::GridSec, "gridSec");
        FieldMappings_["sensorConf"] = Utf8ToTMetricConfFieldParser(Parser_, &TServiceConfig::MetricConf, "sensorConf");
        FieldMappings_["sensorNameLabel"] = Utf8FieldParser(Parser_, &TServiceConfig::MetricNameLabel, "sensorNameLabel");
        FieldMappings_["shardSettings"] = ShardSettingsFieldParser(Parser_, &TServiceConfig::ShardSettings, "shardSettings");
        FieldMappings_["labels"] = Utf8FieldParser(Parser_, &TServiceConfig::Labels, "labels");
    }
};

auto UnwrapOne = ParseOne<NModel::TServiceConfig, TServiceParser>;
auto UnwrapReadTableResult = ParseReadTableResult<NModel::TServiceConfig, TServiceParser>;


class TServiceDao: public IServiceConfigDao, private TDaoBase {
public:
    TServiceDao(TString path, std::shared_ptr<TTableClient> client, TMetricRegistry& registry)
        : TDaoBase{std::move(client), registry}
        , TablePath_{std::move(path)}
    {
        Init();
    }

    const TString& Name() const override {
        static const TString NAME{"ServicesDao"};
        return NAME;
    }

    const TVector<TStringBuf>& Methods() const override {
        static const TVector<TStringBuf> METHODS{
            TStringBuf("createSchema"),
            TStringBuf("dropSchema"),
            TStringBuf("deleteByProject"),
            TStringBuf("insert"),
            TStringBuf("findByProject"),
            TStringBuf("findByName"),
            TStringBuf("findAll"),
        };

        return METHODS;
    }

    TAsyncVoid CreateTable() override {
        const auto req = RawQueries_.at(SERVICE_CREATE_TABLE);

        return Execute<TAsyncStatus>(*Client_, [req] (TSession session) {
            return session.ExecuteSchemeQuery(req);
        }).Apply(CompleteStatus(CounterFor("createSchema")));
    }

    TAsyncVoid DropTable() override {
        return Execute<TAsyncStatus>(*Client_, [=] (TSession session) {
            return session.DropTable(TablePath_);
        }).Apply(CompleteStatus(CounterFor("dropSchema")));
    }

    TAsyncVoid Delete(const TString& serviceId, const TString& projectId) override {
        const auto req = RawQueries_.at(SERVICE_DELETE_WITH_PROJECT);
        auto params = TParamsBuilder{}
                .AddParam("$id").Utf8(serviceId).Build()
                .AddParam("$projectId").Utf8(projectId).Build()
                .AddParam("$folderId").Utf8("").Build()
            .Build();

        return Client_->RetryOperation<TDataQueryResult>([=] (NYdb::NTable::TSession s) {
            return ExecutePrepared(std::move(s), req, params);
        }).Apply(CompleteStatus(CounterFor("deleteByProject")));
    }

    TAsyncVoid Insert(const NModel::TServiceConfig& model) override {
        auto req = RawQueries_.at(SERVICE_INSERT);

        auto params = TParamsBuilder{}
                .AddParam("$id").Utf8(model.Id).Build()
                .AddParam("$version").Int32(model.Version).Build()
                .AddParam("$createdAt").Int64(model.CreatedAt.MilliSeconds()).Build()
                .AddParam("$createdBy").Utf8(model.CreatedBy).Build()
                .AddParam("$interval").Int32(model.Interval.Seconds()).Build()
                .AddParam("$gridSec").Int32(model.GridSec).Build()
                .AddParam("$name").Utf8(model.Name).Build()
                .AddParam("$description").Utf8(model.Description).Build()
                .AddParam("$serviceProvider").Utf8(model.ServiceProvider).Build()
                .AddParam("$projectId").Utf8(model.ProjectId).Build()
                .AddParam("$folderId").Utf8("").Build()
                .AddParam("$sensorConf").Utf8(model.MetricConf.ToString()).Build()
                .AddParam("$sensorNameLabel").Utf8(model.MetricNameLabel).Build()
                .AddParam("$shardSettings").Utf8(model.ShardSettings.ToJsonStr()).Build()
                .AddParam("$labels").Utf8(model.Labels).Build()
                .AddParam("$updatedAt").Int64(model.UpdatedAt.MilliSeconds()).Build()
                .AddParam("$updatedBy").Utf8(model.UpdatedBy).Build()
                // TODO: remove these stubs
                .AddParam("$addTsArgs").Bool(false).Build()
                .AddParam("$path").Utf8(TString{}).Build()
                .AddParam("$port").Int32(0).Build()
                .AddParam("$tvmDestId").Utf8(TString{}).Build()
                .AddParam("$sensorsTtlDays").Int32(0).Build()
            .Build();

        return Client_->RetryOperation<TDataQueryResult>([=] (NYdb::NTable::TSession s) {
            return ExecutePrepared(std::move(s), req, params);
        }).Apply(CompleteStatus(CounterFor("insert")));
    }

    TAsyncServiceConfig GetByProject(const TString& serviceId, const TString& projectId) override {
        auto q = RawQueries_.at(SERVICE_FIND_WITH_PROJECT);
        auto params = TParamsBuilder{}
                .AddParam("$projectId").Utf8(projectId).Build()
                .AddParam("$folderId").Utf8("").Build()
                .AddParam("$id").Utf8(serviceId).Build()
            .Build();

        auto p = NewPromise<std::optional<NModel::TServiceConfig>>();
        ReadDataWithRetries(*Client_, q, std::move(params), p, UnwrapOne, CounterFor("findByProject"));
        return p.GetFuture();
    }

    TAsyncServiceConfig GetByName(const TString& serviceName, const TString& projectId) override {
        auto q = RawQueries_.at(SERVICE_FIND_BY_NAME);
        auto params = TParamsBuilder{}
                .AddParam("$projectId").Utf8(projectId).Build()
                .AddParam("$folderId").Utf8("").Build()
                .AddParam("$name").Utf8(serviceName).Build()
            .Build();

        auto p = NewPromise<std::optional<NModel::TServiceConfig>>();
        ReadDataWithRetries(*Client_, q, std::move(params), p, UnwrapOne, CounterFor("findByName"));
        return p.GetFuture();
    }

    TAsyncServiceConfigs GetAll() override {
        return ReadTable<NModel::TServiceConfig>(*Client_, TablePath_, UnwrapReadTableResult, CounterFor("findAll"));
    }

private:
    THashMap<TStringBuf, TString> LoadQueries() override {
        THashMap<TStringBuf, TString> result;

        auto getOne = [&] (TStringBuf key) {
            const auto resourceKey = TString{key} + ".yql";
            auto resource = NResource::Find(resourceKey);
            Y_ENSURE(!resource.empty(), "Resource " << resourceKey << " is empty");

            SubstGlobal(resource, TStringBuf("${service.table.path}"), TablePath_);
            result.emplace(key, resource);
        };

        getOne(SERVICE_CREATE_TABLE);
        getOne(SERVICE_FIND_WITH_PROJECT);
        getOne(SERVICE_FIND_BY_NAME);
        getOne(SERVICE_DELETE_FOR_PROJECT);
        getOne(SERVICE_DELETE_WITH_PROJECT);
        getOne(SERVICE_INSERT);

        return result;
    }

private:
    const TString TablePath_;
};

} // namespace

IServiceConfigDaoPtr CreateYdbServiceDao(TString table, std::shared_ptr<TTableClient> client, TMetricRegistry& registry) {
    return ::MakeIntrusive<TServiceDao>(std::move(table), std::move(client), registry);
}


} // namespace NSolomon::NDb
